<?php
/**
 * Plugin Name: Agentic Press
 * Plugin URI: https://onestep.quest/agentic-press/
 * Description: Agentic Press plugin for managing author-category assignments and content generation with AI integration.
 * Version: 1.0.0
 * Author: One Step Quest
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: agentic-press
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define constants
define('AP_VERSION', '1.0.0');
define('AP_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('AP_PLUGIN_URL', plugin_dir_url(__FILE__));
define('AP_TABLE_CATEGORIES', 'ap_categories');
define('AP_TABLE_AUTHORS_CATEGORIES', 'ap_authors_categories');

// Include required files
require_once AP_PLUGIN_DIR . 'includes/class-ap-db.php';
require_once AP_PLUGIN_DIR . 'includes/class-ap-admin.php';
require_once AP_PLUGIN_DIR . 'includes/class-ap-user-fields.php';
require_once AP_PLUGIN_DIR . 'includes/class-ap-api.php';
require_once AP_PLUGIN_DIR . 'includes/class-ap-frontend.php';
require_once AP_PLUGIN_DIR . 'includes/class-ap-settings.php';

/**
 * Main plugin class
 */
class Agentic_Press {

    /**
     * Instance of the plugin
     */
    private static $instance = null;

    /**
     * Get instance of the plugin
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        add_action('plugins_loaded', array($this, 'load_textdomain'));
        add_action('init', array($this, 'init'));
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Suppress any output during activation
        ob_start();
        AP_DB::create_tables();
        $this->install_api_files();
        ob_end_clean();
    }

    /**
     * Install API files to web root
     */
    private function install_api_files() {
        $plugin_api_dir = AP_PLUGIN_DIR . 'api/';
        $web_root_api_dir = ABSPATH . 'api/';

        // Create API directory in web root if it doesn't exist
        if (!file_exists($web_root_api_dir)) {
            if (!wp_mkdir_p($web_root_api_dir)) {
                error_log('AP: Failed to create API directory in web root');
                return;
            } else {
                // Set proper permissions on the directory
                if (!chmod($web_root_api_dir, 0755)) {
                    error_log('AP: Failed to set permissions on API directory');
                } else {
                    error_log('AP: Set permissions 0755 on API directory');
                }
            }
        }

        // Copy API files
        $files_to_copy = array('api.php', 'config.php', 'fix_metadata.php', '.htaccess');
        foreach ($files_to_copy as $file) {
            $source = $plugin_api_dir . $file;
            $destination = $web_root_api_dir . $file;

            if (file_exists($source)) {
                if (!copy($source, $destination)) {
                    error_log("AP: Failed to copy $file to web root");
                } else {
                    error_log("AP: Successfully copied $file to web root");
                    // Set proper permissions for web server access
                    if (!chmod($destination, 0644)) {
                        error_log("AP: Failed to set permissions on $destination");
                    } else {
                        error_log("AP: Set permissions 0644 on $destination");
                    }
                }
            } else {
                error_log("AP: Source file $file not found in plugin directory");
            }
        }

        // Flush rewrite rules to ensure API endpoints work
        flush_rewrite_rules();
    }

    /**
     * Uninstall API files from web root
     */
    private function uninstall_api_files() {
        $web_root_api_dir = ABSPATH . 'api/';

        // Remove API files
        $files_to_remove = array('api.php', 'config.php', 'fix_metadata.php', '.htaccess');
        foreach ($files_to_remove as $file) {
            $file_path = $web_root_api_dir . $file;
            if (file_exists($file_path)) {
                if (!unlink($file_path)) {
                    error_log("AP: Failed to remove $file from web root");
                } else {
                    error_log("AP: Successfully removed $file from web root");
                }
            }
        }

        // Remove API directory if empty
        if (file_exists($web_root_api_dir) && count(scandir($web_root_api_dir)) == 2) { // Only . and ..
            rmdir($web_root_api_dir);
            error_log('AP: Removed empty API directory from web root');
        }

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Cleanup if needed
        $this->uninstall_api_files();
    }

    /**
     * Load text domain
     */
    public function load_textdomain() {
        load_plugin_textdomain('autopress', false, dirname(plugin_basename(__FILE__)) . '/languages/');
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Initialize components
        new AP_Admin();
        new AP_User_Fields();
        new AP_API();
        new AP_Frontend();
    }
}

// Initialize the plugin
Agentic_Press::get_instance();