<?php
ini_set('error_log', 'api_debug.log');
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
ob_end_flush();

// Set UTF-8 encoding
ini_set('default_charset', 'UTF-8');
mb_internal_encoding('UTF-8');

// Load WordPress and config
// require_once $_SERVER['DOCUMENT_ROOT'] . '/wp-load.php'; // Removed to reduce resource usage
require_once 'config.php';

// Function to recursively fetch sub-categories
function fetch_subcategories($wp, $parent_id, $WP_TERMS, $WP_TERM_TAXONOMY) {
    $sub_sql = "SELECT $WP_TERMS.term_id, $WP_TERMS.name, $WP_TERMS.slug, $WP_TERM_TAXONOMY.description FROM $WP_TERMS " .
               "JOIN $WP_TERM_TAXONOMY ON $WP_TERMS.term_id = $WP_TERM_TAXONOMY.term_id " .
               "WHERE $WP_TERM_TAXONOMY.taxonomy = 'category' AND $WP_TERM_TAXONOMY.parent = " . (int)$parent_id;
    $sub_result = $wp->query($sub_sql);
    $subcats = [];
    
    if ($sub_result) {
        while ($sub = $sub_result->fetch_assoc()) {
            $subcat = [
                'id' => (int)$sub['term_id'],
                'name' => $sub['name'],
                'slug' => $sub['slug'],
                'description' => $sub['description'],
            ];
            // Recursively fetch sub-categories of this sub-category
            $subcat['subcategories'] = fetch_subcategories($wp, $sub['term_id'], $WP_TERMS, $WP_TERM_TAXONOMY);
            $subcats[] = $subcat;
        }
    }
    
    return $subcats;
}


// === ENHANCED ERROR LOGGING ===
$DEBUG = false; // Enable detailed logging
function error_logger($msg, $level = 'INFO', $context = []) {
	global $DEBUG;
	if ($DEBUG) {
		$timestamp = date('Y-m-d H:i:s');
		$request_id = $_SERVER['REQUEST_ID'] ?? uniqid('req_', true);
		$log_entry = [
			'timestamp' => $timestamp,
			'request_id' => $request_id,
			'level' => $level,
			'message' => $msg,
			'uri' => $_SERVER['REQUEST_URI'] ?? 'unknown',
			'method' => $_SERVER['REQUEST_METHOD'] ?? 'unknown',
			'context' => $context
		];
		$log_line = json_encode($log_entry, JSON_UNESCAPED_UNICODE) . "\n";
		file_put_contents(__DIR__ . '/api_debug.log', $log_line, FILE_APPEND);
		error_log("[$level] $msg");
	}
}

function create_error_response($error_code, $message, $details = null, $context = []) {
	$timestamp = date('Y-m-d H:i:s');
	$request_id = $_SERVER['REQUEST_ID'] ?? uniqid('req_', true);

	$response = [
		'success' => false,
		'error' => [
			'code' => $error_code,
			'message' => $message,
			'timestamp' => $timestamp,
			'request_id' => $request_id
		]
	];

	if ($details) {
		$response['error']['details'] = $details;
	}

	if (!empty($context)) {
		$response['error']['context'] = $context;
	}

	error_logger($message, 'ERROR', array_merge($context, [
		'error_code' => $error_code,
		'details' => $details
	]));

	return $response;
}

// Recursively ensure all strings are valid UTF-8
function utf8ize($mixed) {
	if (is_array($mixed)) {
		foreach ($mixed as $k => $v) {
			$mixed[$k] = utf8ize($v);
		}
	} else if (is_string($mixed)) {
		return mb_convert_encoding($mixed, 'UTF-8', 'UTF-8');
	}
	return $mixed;
}

// Recursively convert smart quotes to regular quotes
function sanitize_quotes($mixed) {
	$search = [
		"\xE2\x80\x9C", // “
		"\xE2\x80\x9D", // ”
		"\xE2\x80\x98", // ‘
		"\xE2\x80\x99", // ’
		'“', '”', '‘', '’'
	];
	$replace = ['"', '"', "'", "'", '"', '"', "'", "'"];
	if (is_array($mixed)) {
		foreach ($mixed as $k => $v) {
			$mixed[$k] = sanitize_quotes($v);
		}
	} else if (is_string($mixed)) {
		return str_replace($search, $replace, $mixed);
	}
	return $mixed;
}

// Helper function to build Agentic Press category structure recursively
function build_autopress_category_structure($wp, $category, $categories_table, $authors_table) {
	global $WP_PREFIX;
	
	// Get authors assigned to this category
	$users_table = $WP_PREFIX . 'users';
	$authors_sql = "SELECT ac.author_id, u.display_name
					FROM $authors_table ac
					JOIN $users_table u ON ac.author_id = u.ID
					WHERE ac.category_id = ?
					ORDER BY u.display_name";
	$authors_stmt = $wp->prepare($authors_sql);
	$authors_stmt->bind_param('i', $category['id']);
	$authors_stmt->execute();
	$authors_result = $authors_stmt->get_result();

	// Build authors structure - all authors go to both prod and lab environments (matching original JSON)
	$prod_authors = [];
	$lab_authors = [];
	while ($author = $authors_result->fetch_assoc()) {
		$author_data = [
			'role' => 'author',
			'name' => $author['display_name'],
			'id' => (int)$author['author_id']
		];
		$prod_authors[] = $author_data;
		$lab_authors[] = $author_data;
	}
	$authors_stmt->close();

	$envs = [];
	if (!empty($prod_authors)) {
		$envs[] = ['env' => 'prod', 'authors' => $prod_authors];
		$envs[] = ['env' => 'lab', 'authors' => $lab_authors];
	}

	// Get subcategories
	$sub_sql = "SELECT * FROM $categories_table WHERE parent_id = ? AND is_active = 1 ORDER BY name";
	$sub_stmt = $wp->prepare($sub_sql);
	$sub_stmt->bind_param('i', $category['id']);
	$sub_stmt->execute();
	$sub_result = $sub_stmt->get_result();

	$subcategories_structure = [];
	while ($subcategory = $sub_result->fetch_assoc()) {
		$subcategories_structure[] = build_autopress_category_structure($wp, $subcategory, $categories_table, $authors_table);
	}
	$sub_stmt->close();

	// Build the complete category structure
	return [
		'id' => (int)$category['term_id'],
		'term_id' => (int)$category['term_id'],
		'category' => $category['category_slug'],
		'name' => $category['name'],
		'slug' => $category['category_slug'],
		'description' => $category['description'],
		'system_prompt' => $category['system_prompt'],
		'user_prompt' => $category['user_prompt'],
		'llm_provider' => $category['llm_provider'],
		'model' => $category['model'],
		'temperature' => (float)$category['temperature'],
		'top_p' => (float)$category['top_p'],
		'max_tokens' => (int)$category['max_tokens'],
		'include_base_system_prompt' => (bool)$category['include_base_system_prompt'],
		'authors' => ['envs' => $envs],
		'sub_categories' => $subcategories_structure
	];
}

// === GLOBAL CONFIGURATION ===
// Load sensitive credentials and config from config.php
require_once __DIR__ . '/config.php';

// === AUTHENTICATION ===
// Example: expects a 'tokens' table with columns: token (string), user_id (int), expires_at (datetime)
function get_bearer_token() {
	$headers = getallheaders();
	if (isset($headers['Authorization'])) {
		if (preg_match('/Bearer\s(\S+)/', $headers['Authorization'], $matches)) {
			return $matches[1];
		}
	}
	return null;
}

function validate_token($token) {
	if (!$token) return false;

	global $TOKENS_TABLE;
	try {
		$db = get_mysql_connection();
		$stmt = $db->prepare("SELECT user_id FROM $TOKENS_TABLE WHERE token = ? AND (expires_at IS NULL OR expires_at > NOW()) LIMIT 1");
		if (!$stmt) return false;
		$stmt->bind_param('s', $token);
		$stmt->execute();
		$stmt->store_result();
		$valid = $stmt->num_rows > 0;
		$stmt->close();
		$db->close();
		return $valid;
	} catch (Exception $e) {
		error_logger("Token validation failed: " . $e->getMessage());
		return false;
	}
}

// Call this at the top of any endpoint that requires authentication
function require_auth() {
	$token = get_bearer_token();
	if (!validate_token($token)) {
		http_response_code(401);
	    echo json_encode(utf8ize(['error' => 'Unauthorized']), JSON_UNESCAPED_UNICODE);
		exit;
	}
}
function get_mysql_connection() {
	global $MYSQL_HOST, $MYSQL_USER, $MYSQL_PASS, $MYSQL_DB;
	// SECURITY: Removed logging of database credentials
	$mysqli = new mysqli($MYSQL_HOST, $MYSQL_USER, $MYSQL_PASS, $MYSQL_DB);
	if ($mysqli->connect_errno) {
		$error_msg = "MySQL connection failed: " . $mysqli->connect_error;
		error_logger($error_msg);
		throw new Exception($error_msg);
	}
	$mysqli->set_charset('utf8mb4');
	// SECURITY: Removed success logging that could confirm valid credentials
	return $mysqli;
}

function get_wp_connection() {
	global $WP_HOST, $WP_USER, $WP_PASS, $WP_DB, $WP_PREFIX;
	$mysqli = new mysqli($WP_HOST, $WP_USER, $WP_PASS, $WP_DB);
	if ($mysqli->connect_errno) {
		http_response_code(500);
	    echo json_encode(utf8ize(['error' => 'WordPress DB connection failed']), JSON_UNESCAPED_UNICODE);
		exit;
	}
	$mysqli->set_charset('utf8mb4');
	return $mysqli;
}

// === BASIC ROUTING ===
header('Content-Type: application/json');
$method = $_SERVER['REQUEST_METHOD'];
$uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$query = $_GET;

// Remove any base path if needed (e.g., /api/)
$base = '/api';
if (strpos($uri, $base) === 0) {
	$uri = substr($uri, strlen($base));
}

// Route requests
switch (true) {
	// GET /version
	case $method === 'GET' && $uri === '/version':
		echo json_encode(['version' => '1.4', 'updated' => '2025-09-23']);
		break;

	// POST /login
	case $method === 'POST' && preg_match('#^/login/?$#', $uri):
		// Accept JSON body: {"username": "...", "password": "..."}
		$input = json_decode(file_get_contents('php://input'), true);
		if (!isset($input['username'], $input['password'])) {
			http_response_code(400);
	        echo json_encode(utf8ize(['error' => 'Username and password required']), JSON_UNESCAPED_UNICODE);
			break;
		}
		$db = get_mysql_connection();
		// Example: users table with columns id, username, password_hash
        global $USERS_TABLE;
        $stmt = $db->prepare("SELECT id, password_hash FROM $USERS_TABLE WHERE username = ? LIMIT 1");
		$stmt->bind_param('s', $input['username']);
		$stmt->execute();
		$stmt->store_result();
		if ($stmt->num_rows === 0) {
			http_response_code(401);
	        echo json_encode(utf8ize(['error' => 'Invalid credentials']), JSON_UNESCAPED_UNICODE);
			$stmt->close();
			$db->close();
			break;
		}
		$stmt->bind_result($user_id, $password_hash);
		$stmt->fetch();
		if (!password_verify($input['password'], $password_hash)) {
			http_response_code(401);
	        echo json_encode(utf8ize(['error' => 'Invalid credentials']), JSON_UNESCAPED_UNICODE);
			$stmt->close();
			$db->close();
			break;
		}
		$stmt->close();
		// Generate token
		$token = bin2hex(random_bytes(32));
		$expires = date('Y-m-d H:i:s', strtotime('+1 day'));
		global $TOKENS_TABLE;
		// Delete any existing token for this user to avoid duplicate entry error
		$del_stmt = $db->prepare("DELETE FROM $TOKENS_TABLE WHERE user_id = ?");
		$del_stmt->bind_param('i', $user_id);
		$del_stmt->execute();
		$del_stmt->close();
		// Now insert the new token
		$stmt2 = $db->prepare("INSERT INTO $TOKENS_TABLE (token, user_id, expires_at) VALUES (?, ?, ?)");
		$stmt2->bind_param('sis', $token, $user_id, $expires);
		$stmt2->execute();
		$stmt2->close();
		$db->close();
		echo json_encode(utf8ize(['token' => $token, 'expires_at' => $expires]), JSON_UNESCAPED_UNICODE);
		break;

	// POST /create_user
	case $method === 'POST' && preg_match('#^/create_user/?$#', $uri):
		require_auth();
		$input = json_decode(file_get_contents('php://input'), true);
		if (!isset($input['username'], $input['email'], $input['password'], $input['name'])) {
			echo json_encode(utf8ize(['error' => 'Username, email, password, and name required']), JSON_UNESCAPED_UNICODE);
			break;
		}
		$username = $input['username'];
		$email = $input['email'];
		$password = $input['password'];
		$name = $input['name'];
		$role = $input['role'] ?? 'author';
		$category_ownership = $input['category_ownership'] ?? []; // Array of category IDs
		
		// Load WordPress only for user creation
		$wp_load_path = $_SERVER['DOCUMENT_ROOT'] . '/wp-load.php';
		if (!file_exists($wp_load_path)) {
			http_response_code(500);
			echo json_encode(utf8ize(['error' => 'WordPress not available for user creation']), JSON_UNESCAPED_UNICODE);
			break;
		}
		require_once $wp_load_path;
		
		$user_id = wp_create_user($username, $password, $email);
		if (is_wp_error($user_id)) {
			echo json_encode(utf8ize(['error' => $user_id->get_error_message()]), JSON_UNESCAPED_UNICODE);
		} else {
			$user = new WP_User($user_id);
			$user->set_role($role);
			$name_parts = explode(' ', $name, 2);
			$first_name = $name_parts[0];
			$last_name = isset($name_parts[1]) ? $name_parts[1] : '';
			wp_update_user(['ID' => $user_id, 'display_name' => $name, 'first_name' => $first_name, 'last_name' => $last_name]);
			
			// Store category ownership if provided
			if (!empty($category_ownership) && is_array($category_ownership)) {
				update_user_meta($user_id, 'author_category_ownership', $category_ownership);
			}
			
			echo json_encode(utf8ize(['success' => true, 'user_id' => $user_id]), JSON_UNESCAPED_UNICODE);
		}
		break;
	
    // GET /categories
    case $method === 'GET' && preg_match('#^/categories/?$#', $uri):
        error_log('Entered /categories endpoint');
        require_auth();
        // Fetch all top-level categories from WordPress DB
        global $WP_TERMS, $WP_TERM_TAXONOMY;
        $includeSub = isset($query['includeSub']) && $query['includeSub'] === 'true';
        $wp = get_wp_connection();
        if (!$wp) {
            error_log('WordPress DB connection failed');
	        echo json_encode(utf8ize(['error' => 'DB connection failed']), JSON_UNESCAPED_UNICODE);
            exit;
        }
        // Get top-level categories (parent = 0)
        $sql = "SELECT $WP_TERMS.term_id, $WP_TERMS.name, $WP_TERMS.slug, $WP_TERM_TAXONOMY.description FROM $WP_TERMS " .
            "JOIN $WP_TERM_TAXONOMY ON $WP_TERMS.term_id = $WP_TERM_TAXONOMY.term_id " .
            "WHERE $WP_TERM_TAXONOMY.taxonomy = 'category' AND $WP_TERM_TAXONOMY.parent = 0";
        $result = $wp->query($sql);
        if (!$result) {
            error_log('Category query failed: ' . $wp->error);
	        echo json_encode(utf8ize(['error' => 'Query failed', 'details' => $wp->error]), JSON_UNESCAPED_UNICODE);
            exit;
        }

        $categories = [];
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $cat = [
                    'id' => (int)$row['term_id'],
                    'name' => $row['name'],
                    'slug' => $row['slug'],
                    'description' => $row['description'],
                ];
                if ($includeSub) {
					// Fetch sub-categories recursively for this category
                    $cat['subcategories'] = fetch_subcategories($wp, $row['term_id'], $WP_TERMS, $WP_TERM_TAXONOMY);
                }
                $categories[] = $cat;
            }
        }
		$wp->close();
		$categories = sanitize_quotes($categories);
		$categories = utf8ize($categories);
		$json = json_encode(utf8ize(['categories' => $categories]), JSON_UNESCAPED_UNICODE);
		if ($json === false) {
			error_log('json_encode error: ' . json_last_error_msg());
			echo json_encode(utf8ize(['error' => 'Failed to encode categories', 'details' => json_last_error_msg()]), JSON_UNESCAPED_UNICODE);
		} else {
			echo $json;
			flush();
		}
		break;

	// POST /update_user
	case $method === 'POST' && preg_match('#^/update_user/?$#', $uri):
		require_auth();
		$input = json_decode(file_get_contents('php://input'), true);
		if (!isset($input['user_id'], $input['name'])) {
			echo json_encode(utf8ize(['error' => 'user_id and name required']), JSON_UNESCAPED_UNICODE);
			break;
		}
		$user_id = $input['user_id'];
		$name = $input['name'];
		$category_ownership = $input['category_ownership'] ?? null; // Optional category ownership update
		$name_parts = explode(' ', $name, 2);
		$first_name = $name_parts[0];
		$last_name = isset($name_parts[1]) ? $name_parts[1] : '';
		
		// Load WordPress only for user update
		$wp_load_path = $_SERVER['DOCUMENT_ROOT'] . '/wp-load.php';
		if (!file_exists($wp_load_path)) {
			http_response_code(500);
			echo json_encode(utf8ize(['error' => 'WordPress not available for user update']), JSON_UNESCAPED_UNICODE);
			break;
		}
		require_once $wp_load_path;
		
		$result = wp_update_user(['ID' => $user_id, 'display_name' => $name, 'first_name' => $first_name, 'last_name' => $last_name]);
		if (is_wp_error($result)) {
			echo json_encode(utf8ize(['error' => $result->get_error_message()]), JSON_UNESCAPED_UNICODE);
		} else {
			// Update category ownership if provided
			if ($category_ownership !== null) {
				if (is_array($category_ownership)) {
					update_user_meta($user_id, 'author_category_ownership', $category_ownership);
				} elseif (empty($category_ownership)) {
					delete_user_meta($user_id, 'author_category_ownership');
				}
			}
			
			echo json_encode(utf8ize(['success' => true, 'user_id' => $user_id]), JSON_UNESCAPED_UNICODE);
		}
		break;

	// GET /user_category_ownership/:user_id
	case $method === 'GET' && preg_match('#^/user_category_ownership/(\d+)$#', $uri, $matches):
		require_auth();
		$user_id = (int)$matches[1];
		
		// Load WordPress to access user meta
		$wp_load_path = $_SERVER['DOCUMENT_ROOT'] . '/wp-load.php';
		if (!file_exists($wp_load_path)) {
			http_response_code(500);
			echo json_encode(utf8ize(['error' => 'WordPress not available']), JSON_UNESCAPED_UNICODE);
			break;
		}
		require_once $wp_load_path;
		
		$category_ownership = get_user_meta($user_id, 'author_category_ownership', true);
		if (empty($category_ownership)) {
			$category_ownership = [];
		}
		
		echo json_encode(utf8ize([
			'user_id' => $user_id,
			'category_ownership' => $category_ownership
		]), JSON_UNESCAPED_UNICODE);
		break;
	
    // GET /categories/:category
	case $method === 'GET' && preg_match('#^/categories/([^/]+)$#', $uri, $matches):
	error_logger("Entered /categories/{$matches[1]} endpoint");
		require_auth();
		global $WP_TERMS, $WP_TERM_TAXONOMY;
		$category_param = $matches[1];
		$wp = get_wp_connection();

		// Try to find the category by slug, then name (case-insensitive), then ID
		$cat = null;
		if (is_numeric($category_param)) {
			$cat_sql = "SELECT t.term_id, t.name, t.slug, tt.description FROM $WP_TERMS t JOIN $WP_TERM_TAXONOMY tt ON t.term_id = tt.term_id WHERE tt.taxonomy = 'category' AND t.term_id = ? LIMIT 1";
			error_log("Trying by term_id: $category_param");
			$stmt = $wp->prepare($cat_sql);
			$stmt->bind_param('i', $category_param);
			$stmt->execute();
			$result = $stmt->get_result();
			if ($result && $result->num_rows > 0) {
				$cat = $result->fetch_assoc();
				error_log("Found by term_id: " . json_encode($cat));
			} else {
				error_log("No category found by term_id");
			}
			$stmt->close();
		} else {
			// Try slug first
			$cat_sql = "SELECT t.term_id, t.name, t.slug, tt.description FROM $WP_TERMS t JOIN $WP_TERM_TAXONOMY tt ON t.term_id = tt.term_id WHERE tt.taxonomy = 'category' AND t.slug = ? LIMIT 1";
			error_logger("Trying by slug: $category_param");
			$stmt = $wp->prepare($cat_sql);
			if (!$stmt) {
				error_logger("Prepare failed: " . $wp->error);
			}
			$stmt->bind_param('s', $category_param);
			$stmt->execute();
			$result = $stmt->get_result();
			if ($result && $result->num_rows > 0) {
				$cat = $result->fetch_assoc();
				error_logger("fetch_assoc() result: " . var_export($cat, true));
				error_logger("Found by slug: " . json_encode($cat));
			} else {
				error_logger("No category found by slug. SQL error: " . $wp->error);
			}
			$stmt->close();
			// If not found, try name (case-insensitive)
			if (!$cat) {
				$cat_sql = "SELECT t.term_id, t.name, t.slug, tt.description FROM $WP_TERMS t JOIN $WP_TERM_TAXONOMY tt ON t.term_id = tt.term_id WHERE tt.taxonomy = 'category' AND LOWER(t.name) = LOWER(?) LIMIT 1";
				error_logger("Trying by name: $category_param");
				$stmt = $wp->prepare($cat_sql);
				$stmt->bind_param('s', $category_param);
				$stmt->execute();
				$result = $stmt->get_result();
				if ($result && $result->num_rows > 0) {
					$cat = $result->fetch_assoc();
					error_logger("Found by name: " . json_encode($cat));
				} else {
					error_logger("No category found by name");
				}
				$stmt->close();
			}
		}
		if (!$cat) {
			error_logger("Category not found for param: $category_param");
			http_response_code(404);
	        echo json_encode(utf8ize(['error' => 'Category not found']), JSON_UNESCAPED_UNICODE);
			$wp->close();
			break;
		}

		// Fetch sub-categories recursively
		$cat['subcategories'] = fetch_subcategories($wp, $cat['term_id'], $WP_TERMS, $WP_TERM_TAXONOMY);
		$cat = sanitize_quotes($cat);
		$cat = utf8ize($cat);
		$json = json_encode(utf8ize(['category' => $cat]), JSON_UNESCAPED_UNICODE);
		if ($json === false) {
			error_logger('json_encode error: ' . json_last_error_msg());
			echo json_encode(utf8ize(['error' => 'Failed to encode category', 'details' => json_last_error_msg()]), JSON_UNESCAPED_UNICODE);
		} else {
			echo $json;
		}
		break;

    // GET /posts
	case $method === 'GET' && preg_match('#^/posts/?$#', $uri):
		error_logger(date('c') . " - /posts route hit");
		require_auth();
		global $WP_DB, $WP_POSTS, $WP_TERM_TAXONOMY, $WP_TERMS, $WP_POSTMETA;
		$lookback = isset($query['lookback']) && is_numeric($query['lookback']) ? (int)$query['lookback'] : 7;
		$limit = isset($query['limit']) && is_numeric($query['limit']) ? (int)$query['limit'] : 10;
		$category = isset($query['category']) ? $query['category'] : null;
		$without_images = isset($query['without_images']) && $query['without_images'] === 'true';
		$status = isset($query['status']) ? $query['status'] : 'publish';
		$wp = get_wp_connection();

		$debug = [];
		$posts = [];
		
		// Build status condition
		if ($status === 'any') {
			$status_condition = "p.post_status IN ('publish', 'draft', 'pending', 'private')";
		} else {
			$status_condition = "p.post_status = '" . $wp->real_escape_string($status) . "'";
		}
		$debug['status'] = $status;
		$debug['status_condition'] = $status_condition;
		if ($category) {
			$debug['category_param'] = $category;
			// Find category term_id, name, slug by slug or ID
			if (is_numeric($category)) {
				$cat_sql = "SELECT term_id, name, slug FROM $WP_TERMS WHERE term_id = ? LIMIT 1";
				$cat_stmt = $wp->prepare($cat_sql);
				$cat_stmt->bind_param('i', $category);
			} else {
				$cat_sql = "SELECT term_id, name, slug FROM $WP_TERMS WHERE slug = ? LIMIT 1";
				$cat_stmt = $wp->prepare($cat_sql);
				$cat_stmt->bind_param('s', $category);
			}
			$debug['cat_sql'] = $cat_sql;
			$cat_stmt->execute();
			$cat_result = $cat_stmt->get_result();
			if (!$cat_result || $cat_result->num_rows === 0) {
				http_response_code(404);
			    echo json_encode(utf8ize(['error' => 'Category not found', 'debug' => $debug]), JSON_UNESCAPED_UNICODE);
				$cat_stmt->close();
				$wp->close();
				break;
			}
			$cat_row = $cat_result->fetch_assoc();
			$term_id = (int)$cat_row['term_id'];
			$cat_name = $cat_row['name'];
			$cat_slug = $cat_row['slug'];
			$cat_stmt->close();

			// Get post IDs for this category (from term_relationships)
			$rel_sql = "SELECT object_id FROM {$WP_PREFIX}term_relationships WHERE term_taxonomy_id = (SELECT term_taxonomy_id FROM $WP_TERM_TAXONOMY WHERE term_id = ? AND taxonomy = 'category' LIMIT 1)";
			$rel_stmt = $wp->prepare($rel_sql);
			$rel_stmt->bind_param('i', $term_id);
			$debug['rel_sql'] = $rel_sql;
			$rel_stmt->execute();
			$rel_result = $rel_stmt->get_result();
			$post_ids = [];
			if ($rel_result) {
				while ($rel = $rel_result->fetch_assoc()) {
					$post_ids[] = (int)$rel['object_id'];
				}
			}
			$debug['post_ids'] = $post_ids;
			$rel_stmt->close();

			if (count($post_ids) > 0) {
				// Build placeholders for IN clause
				$in = implode(',', array_fill(0, count($post_ids), '?'));
				$types = str_repeat('i', count($post_ids)) . 'ii';
				$base_sql = "SELECT p.ID, p.post_title, p.post_date, p.post_name, p.post_excerpt, p.post_content FROM $WP_POSTS p";
				if ($without_images) {
					$base_sql .= " LEFT JOIN $WP_POSTMETA pm ON p.ID = pm.post_id AND pm.meta_key = '_thumbnail_id'";
				}
				$base_sql .= " WHERE p.ID IN ($in) AND p.post_type = 'post' AND $status_condition AND p.post_date >= DATE_SUB(NOW(), INTERVAL ? DAY)";
				if ($without_images) {
					$base_sql .= " AND pm.meta_id IS NULL";
				}
				$base_sql .= " ORDER BY p.post_date DESC LIMIT ?";
				$sql = $base_sql;
				$stmt = $wp->prepare($sql);
				$params = array_merge($post_ids, [$lookback, $limit]);
				$debug['main_sql'] = $sql;
				$debug['main_sql_params'] = $params;
				$stmt->bind_param($types, ...$params);
				$stmt->execute();
				$result = $stmt->get_result();
				if ($result) {
					while ($row = $result->fetch_assoc()) {
						$post_id = (int)$row['ID'];
						// Get tags for this post
						$tag_sql = "SELECT t.term_id, t.name, t.slug FROM {$WP_PREFIX}term_relationships r JOIN $WP_TERM_TAXONOMY tt ON r.term_taxonomy_id = tt.term_taxonomy_id JOIN $WP_TERMS t ON tt.term_id = t.term_id WHERE r.object_id = ? AND tt.taxonomy = 'post_tag'";
						$tag_stmt = $wp->prepare($tag_sql);
						$tags = [];
						if ($tag_stmt) {
							$tag_stmt->bind_param('i', $post_id);
							$tag_stmt->execute();
							$tag_result = $tag_stmt->get_result();
							if ($tag_result) {
								while ($tag_row = $tag_result->fetch_assoc()) {
									$tags[] = [
										'id' => (int)$tag_row['term_id'],
										'name' => $tag_row['name'],
										'slug' => $tag_row['slug']
									];
								}
							}
							$tag_stmt->close();
						}
						$posts[] = [
							'id' => (int)$row['ID'],
							'title' => $row['post_title'],
							'slug' => $row['post_name'],
							'date' => $row['post_date'],
							'excerpt' => $row['post_excerpt'],
							'content' => $row['post_content'],
							'category' => [
								'id' => $term_id,
								'name' => $cat_name,
								'slug' => $cat_slug
							],
							'tags' => $tags
						];
					}
				} else {
					$debug['main_sql_error'] = $stmt->error;
				}
				$stmt->close();
			} else {
				$debug['no_post_ids'] = true;
			}
		} else {
			// No category filter, fetch all
			// For each post, fetch its first category (if any)
			$base_sql = "SELECT p.ID, p.post_title, p.post_date, p.post_name, p.post_excerpt, p.post_content FROM $WP_POSTS p";
			if ($without_images) {
				$base_sql .= " LEFT JOIN $WP_POSTMETA pm ON p.ID = pm.post_id AND pm.meta_key = '_thumbnail_id'";
			}
			$base_sql .= " WHERE p.post_type = 'post' AND $status_condition AND p.post_date >= DATE_SUB(NOW(), INTERVAL ? DAY)";
			if ($without_images) {
				$base_sql .= " AND pm.meta_id IS NULL";
			}
			$base_sql .= " ORDER BY p.post_date DESC LIMIT ?";
			$sql = $base_sql;
			$debug['main_sql'] = $sql;
			$debug['main_sql_params'] = [$lookback, $limit];
			$stmt = $wp->prepare($sql);
			if (!$stmt) {
				$debug['prepare_error'] = $wp->error;
			} else {
				$stmt->bind_param('ii', $lookback, $limit);
				$stmt->execute();
				$result = $stmt->get_result();
				if ($result) {
					while ($row = $result->fetch_assoc()) {
						$post_id = (int)$row['ID'];
						// Get first category for this post
						$cat_sql = "SELECT t.term_id, t.name, t.slug FROM {$WP_PREFIX}term_relationships r JOIN $WP_TERM_TAXONOMY tt ON r.term_taxonomy_id = tt.term_taxonomy_id JOIN $WP_TERMS t ON tt.term_id = t.term_id WHERE r.object_id = ? AND tt.taxonomy = 'category' LIMIT 1";
						$cat_stmt = $wp->prepare($cat_sql);
						if (!$cat_stmt) {
							$debug['cat_stmt_error'] = $wp->error;
						} else {
							$cat_stmt->bind_param('i', $post_id);
							$cat_stmt->execute();
							$cat_result = $cat_stmt->get_result();
							$category_data = null;
							if ($cat_result && $cat_result->num_rows > 0) {
								$cat_row = $cat_result->fetch_assoc();
								$category_data = [
									'id' => (int)$cat_row['term_id'],
									'name' => $cat_row['name'],
									'slug' => $cat_row['slug']
								];
							}
							$cat_stmt->close();
							// Get tags for this post
							$tag_sql = "SELECT t.term_id, t.name, t.slug FROM {$WP_PREFIX}term_relationships r JOIN $WP_TERM_TAXONOMY tt ON r.term_taxonomy_id = tt.term_taxonomy_id JOIN $WP_TERMS t ON tt.term_id = t.term_id WHERE r.object_id = ? AND tt.taxonomy = 'post_tag'";
							$tag_stmt = $wp->prepare($tag_sql);
							$tags = [];
							if ($tag_stmt) {
								$tag_stmt->bind_param('i', $post_id);
								$tag_stmt->execute();
								$tag_result = $tag_stmt->get_result();
								if ($tag_result) {
									while ($tag_row = $tag_result->fetch_assoc()) {
										$tags[] = [
											'id' => (int)$tag_row['term_id'],
											'name' => $tag_row['name'],
											'slug' => $tag_row['slug']
										];
									}
								}
								$tag_stmt->close();
							}
							$posts[] = [
								'id' => $post_id,
								'title' => $row['post_title'],
								'slug' => $row['post_name'],
								'date' => $row['post_date'],
								'excerpt' => $row['post_excerpt'],
								'content' => $row['post_content'],
								'category' => $category_data,
								'tags' => $tags
							];
						}
					}
				} else {
					$debug['main_sql_error'] = $stmt->error;
				}
				$stmt->close();
			}
		}
		$wp->close();
		error_logger(date('c') . " - /posts route end\n" . print_r(['posts' => $posts, 'debug' => $debug], true));
	$sanitized = sanitize_quotes(['posts' => $posts, 'debug' => $debug]);
	$utf8 = utf8ize($sanitized);
	error_logger('Sanitized output: ' . print_r($sanitized, true));
	error_logger('UTF8ized output: ' . print_r($utf8, true));
	$json = json_encode($utf8, JSON_UNESCAPED_UNICODE);
	error_logger('JSON output: ' . $json);
	echo $json;
		break;

	// GET /post/:id
	case $method === 'GET' && preg_match('#^/post/(\d+)$#', $uri, $matches):
		require_auth();
		$post_id = (int)$matches[1];
		global $WP_DB, $WP_POSTS, $WP_TERM_TAXONOMY, $WP_TERMS, $WP_PREFIX;
		$wp = get_wp_connection();
		// Fetch the post (include both published and draft posts for AgenticPress verification)
		$sql = "SELECT ID, post_title, post_date, post_name, post_excerpt, post_content, post_status FROM $WP_POSTS WHERE ID = ? AND post_type = 'post' AND post_status IN ('publish', 'draft') LIMIT 1";
		$stmt = $wp->prepare($sql);
		$stmt->bind_param('i', $post_id);
		$stmt->execute();
		$result = $stmt->get_result();
		if ($result && $result->num_rows > 0) {
			$row = $result->fetch_assoc();
			// Get first category for this post
			$cat_sql = "SELECT t.term_id, t.name, t.slug FROM {$WP_PREFIX}term_relationships r JOIN $WP_TERM_TAXONOMY tt ON r.term_taxonomy_id = tt.term_taxonomy_id JOIN $WP_TERMS t ON tt.term_id = t.term_id WHERE r.object_id = ? AND tt.taxonomy = 'category' LIMIT 1";
			$cat_stmt = $wp->prepare($cat_sql);
			$cat_stmt->bind_param('i', $post_id);
			$cat_stmt->execute();
			$cat_result = $cat_stmt->get_result();
			$category_data = null;
			if ($cat_result && $cat_result->num_rows > 0) {
				$cat_row = $cat_result->fetch_assoc();
				$category_data = [
					'id' => (int)$cat_row['term_id'],
					'name' => $cat_row['name'],
					'slug' => $cat_row['slug']
				];
			}
			$cat_stmt->close();
			$post = [
				'id' => (int)$row['ID'],
				'title' => $row['post_title'],
				'slug' => $row['post_name'],
				'date' => $row['post_date'],
				'excerpt' => $row['post_excerpt'],
				'content' => $row['post_content'],
				'status' => $row['post_status'],
				'category' => $category_data
			];
			$post = sanitize_quotes($post);
			$stmt->close();
			$wp->close();
			echo json_encode(utf8ize(['post' => $post]), JSON_UNESCAPED_UNICODE);
		} else {
			$stmt->close();
			$wp->close();
			http_response_code(404);
			echo json_encode(utf8ize(['error' => 'Post not found']), JSON_UNESCAPED_UNICODE);
		}
		break;

	// POST /post
	case $method === 'POST' && preg_match('#^/post/?$#', $uri):
		require_auth();
		global $WP_POSTS, $WP_TERMS, $WP_TERM_TAXONOMY, $WP_PREFIX;
		$input = json_decode(file_get_contents('php://input'), true);
		$posts = is_array($input) && isset($input[0]) ? $input : [$input];
		$results = [];
		$wp = get_wp_connection();
		foreach ($posts as $post) {
			// Check for required fields - accept either 'category' or 'categories'
			$has_category = isset($post['category']) || isset($post['categories']);
			if (!isset($post['title'], $post['content'], $post['excerpt']) || !$has_category) {
				$results[] = ['success' => false, 'error' => 'title, content, excerpt, and category/categories are required'];
				continue;
			}
			$title = trim($post['title']);
			$content = trim($post['content']);
			$excerpt = trim($post['excerpt']);
			
			// Handle both 'category' (single) and 'categories' (array)
			$categories = [];
			if (isset($post['categories']) && is_array($post['categories'])) {
				$categories = $post['categories'];
			} elseif (isset($post['category'])) {
				$categories = [$post['category']];
			}
			
			$author_id = $post['author_id'] ?? 1;
			if ($title === '' || $content === '' || $excerpt === '' || empty($categories)) {
				$results[] = ['success' => false, 'error' => 'title, content, excerpt, and category/categories cannot be empty'];
				continue;
			}
			
			// Process all categories
			$category_term_ids = [];
			foreach ($categories as $category) {
				if (trim($category) === '') continue;
				
				// Find category term_id by slug or ID
				if (is_numeric($category)) {
					$cat_sql = "SELECT term_id FROM $WP_TERMS WHERE term_id = ? LIMIT 1";
					$cat_stmt = $wp->prepare($cat_sql);
					$cat_stmt->bind_param('i', $category);
				} else {
					$cat_sql = "SELECT term_id FROM $WP_TERMS WHERE slug = ? LIMIT 1";
					$cat_stmt = $wp->prepare($cat_sql);
					$cat_stmt->bind_param('s', $category);
				}
				$cat_stmt->execute();
				$cat_result = $cat_stmt->get_result();
				if (!$cat_result || $cat_result->num_rows === 0) {
					$results[] = ['success' => false, 'error' => "Category '$category' not found"];
					$cat_stmt->close();
					continue 2; // Skip this post
				}
				$cat_row = $cat_result->fetch_assoc();
				$category_term_ids[] = (int)$cat_row['term_id'];
				$cat_stmt->close();
			}
			
			if (empty($category_term_ids)) {
				$results[] = ['success' => false, 'error' => 'No valid categories found'];
				continue;
			}

			// Insert post as draft
			$now = date('Y-m-d H:i:s');
			// Use provided slug if available, otherwise generate from title
			if (isset($post['slug']) && !empty(trim($post['slug']))) {
				$slug = trim($post['slug']);
			} else {
				// Generate slug from title with better handling
				$slug = mb_strtolower($title, 'UTF-8');
				// Remove accents and normalize unicode
				$slug = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $slug);
				// Replace non-alphanumeric characters with hyphens
				$slug = preg_replace('/[^a-z0-9]+/', '-', $slug);
				// Remove multiple consecutive hyphens and trim
				$slug = preg_replace('/-+/', '-', $slug);
				$slug = trim($slug, '-');
				// Ensure it's not empty
				if (empty($slug)) {
					$slug = 'untitled-post';
				}
			}
			$sql = "INSERT INTO $WP_POSTS (post_author, post_date, post_date_gmt, post_content, post_title, post_excerpt, post_status, comment_status, ping_status, post_name, post_type, post_modified, post_modified_gmt) VALUES (?, ?, ?, ?, ?, ?, 'draft', 'open', 'open', ?, 'post', ?, ?)";
			$stmt = $wp->prepare($sql);
			$stmt->bind_param('issssssss', $author_id, $now, $now, $content, $title, $excerpt, $slug, $now, $now);
			if (!$stmt->execute()) {
				$results[] = ['success' => false, 'error' => 'Failed to insert post'];
				$stmt->close();
				continue;
			}
			$post_id = $stmt->insert_id;
			$stmt->close();

			// Assign all categories to the post
			$category_assigned = false;
			foreach ($category_term_ids as $term_id) {
				// Get term_taxonomy_id for this category
				$tt_sql = "SELECT term_taxonomy_id FROM $WP_TERM_TAXONOMY WHERE term_id = ? AND taxonomy = 'category' LIMIT 1";
				$tt_stmt = $wp->prepare($tt_sql);
				$tt_stmt->bind_param('i', $term_id);
				$tt_stmt->execute();
				$tt_result = $tt_stmt->get_result();
				if (!$tt_result || $tt_result->num_rows === 0) {
					$results[] = ['success' => false, 'error' => 'Category taxonomy not found for term_id: ' . $term_id];
					$tt_stmt->close();
					continue 2; // Skip this post
				}
				$tt_row = $tt_result->fetch_assoc();
				$term_taxonomy_id = (int)$tt_row['term_taxonomy_id'];
				$tt_stmt->close();

				// Insert into term_relationships
				$rel_sql = "INSERT INTO {$WP_PREFIX}term_relationships (object_id, term_taxonomy_id) VALUES (?, ?)";
				$rel_stmt = $wp->prepare($rel_sql);
				$rel_stmt->bind_param('ii', $post_id, $term_taxonomy_id);
				if (!$rel_stmt->execute()) {
					$results[] = ['success' => false, 'error' => 'Failed to assign category'];
					$rel_stmt->close();
					continue 2; // Skip this post
				}
				$rel_stmt->close();
				$category_assigned = true;
			}
			
			if (!$category_assigned) {
				$results[] = ['success' => false, 'error' => 'Failed to assign any categories'];
				continue;
			}

			// Handle tags if provided
			if (isset($post['tags_input']) && is_array($post['tags_input'])) {
				error_logger(date('c') . " - Processing tags for post $post_id: " . json_encode($post['tags_input']));
				foreach ($post['tags_input'] as $tag_name) {
					$tag_name = trim($tag_name);
					if ($tag_name === '') continue;

					// Check if tag exists, if not create it
					$tag_check_sql = "SELECT t.term_id FROM $WP_TERMS t INNER JOIN $WP_TERM_TAXONOMY tt ON t.term_id = tt.term_id WHERE t.name = ? AND tt.taxonomy = 'post_tag' LIMIT 1";
					$tag_check_stmt = $wp->prepare($tag_check_sql);
					$tag_check_stmt->bind_param('s', $tag_name);
					$tag_check_stmt->execute();
					$tag_check_result = $tag_check_stmt->get_result();

					if ($tag_check_result && $tag_check_result->num_rows > 0) {
						// Tag exists, get its term_id
						$tag_row = $tag_check_result->fetch_assoc();
						$tag_term_id = (int)$tag_row['term_id'];
					} else {
						// Tag doesn't exist, create it
						// Generate slug from tag name with better handling
						$tag_slug = mb_strtolower($tag_name, 'UTF-8');
						// Remove accents and normalize unicode
						$tag_slug = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $tag_slug);
						// Replace non-alphanumeric characters with hyphens
						$tag_slug = preg_replace('/[^a-z0-9]+/', '-', $tag_slug);
						// Remove multiple consecutive hyphens and trim
						$tag_slug = preg_replace('/-+/', '-', $tag_slug);
						$tag_slug = trim($tag_slug, '-');
						// Ensure it's not empty
						if (empty($tag_slug)) {
							$tag_slug = 'untitled-tag';
						}
						$tag_insert_sql = "INSERT INTO $WP_TERMS (name, slug) VALUES (?, ?)";
						$tag_insert_stmt = $wp->prepare($tag_insert_sql);
						$tag_insert_stmt->bind_param('ss', $tag_name, $tag_slug);
						if ($tag_insert_stmt->execute()) {
							$tag_term_id = $tag_insert_stmt->insert_id;
							// Insert into term_taxonomy
							$tag_tt_sql = "INSERT INTO $WP_TERM_TAXONOMY (term_id, taxonomy) VALUES (?, 'post_tag')";
							$tag_tt_stmt = $wp->prepare($tag_tt_sql);
							$tag_tt_stmt->bind_param('i', $tag_term_id);
							$tag_tt_stmt->execute();
							$tag_tt_stmt->close();
						} else {
							$tag_term_id = null;
						}
						$tag_insert_stmt->close();
					}
					$tag_check_stmt->close();

					// If we have a valid tag_term_id, associate it with the post
					if ($tag_term_id) {
						// Get term_taxonomy_id for this tag
						$tag_tt_sql = "SELECT term_taxonomy_id FROM $WP_TERM_TAXONOMY WHERE term_id = ? AND taxonomy = 'post_tag' LIMIT 1";
						$tag_tt_stmt = $wp->prepare($tag_tt_sql);
						$tag_tt_stmt->bind_param('i', $tag_term_id);
						$tag_tt_stmt->execute();
						$tag_tt_result = $tag_tt_stmt->get_result();
						if ($tag_tt_result && $tag_tt_result->num_rows > 0) {
							$tag_tt_row = $tag_tt_result->fetch_assoc();
							$tag_term_taxonomy_id = (int)$tag_tt_row['term_taxonomy_id'];
							$tag_tt_stmt->close();

							// Insert into term_relationships for the tag
							$tag_rel_sql = "INSERT INTO {$WP_PREFIX}term_relationships (object_id, term_taxonomy_id) VALUES (?, ?) ON DUPLICATE KEY UPDATE object_id = object_id";
							$tag_rel_stmt = $wp->prepare($tag_rel_sql);
							$tag_rel_stmt->bind_param('ii', $post_id, $tag_term_taxonomy_id);
							$tag_rel_stmt->execute();
							$tag_rel_stmt->close();
						} else {
							$tag_tt_stmt->close();
						}
					}
				}
			}

			$results[] = ['success' => true, 'post_id' => $post_id, 'status' => 'draft', 'tags_processed' => isset($post['tags_input']) ? count($post['tags_input']) : 0];
		}
		$wp->close();
		echo json_encode(['results' => $results]);
		break;

	// PUT /post/{id} - Update post (status and/or content)
	case $method === 'PUT' && preg_match('#^/post/(\d+)$#', $uri, $matches):
		require_auth();
		$post_id = (int)$matches[1];
		global $WP_POSTS;
		$input = json_decode(file_get_contents('php://input'), true);

		$wp = get_wp_connection();

		// Check if post exists
		$check_sql = "SELECT post_status, post_content FROM $WP_POSTS WHERE ID = ? AND post_type = 'post' LIMIT 1";
		$check_stmt = $wp->prepare($check_sql);
		$check_stmt->bind_param('i', $post_id);
		$check_stmt->execute();
		$check_result = $check_stmt->get_result();

		if (!$check_result || $check_result->num_rows === 0) {
			$check_stmt->close();
			$wp->close();
			http_response_code(404);
			echo json_encode(utf8ize(['error' => 'Post not found']), JSON_UNESCAPED_UNICODE);
			break;
		}

		$check_row = $check_result->fetch_assoc();
		$current_status = $check_row['post_status'];
		$current_content = $check_row['post_content'];
		$check_stmt->close();

		$updates = [];
		$update_fields = [];

		// Update post status if provided
		if (isset($input['status'])) {
			$new_status = $input['status'];
			$allowed_statuses = ['draft', 'publish', 'pending', 'private'];
			if (!in_array($new_status, $allowed_statuses)) {
				$wp->close();
				http_response_code(400);
				echo json_encode(utf8ize(['error' => 'Invalid status. Allowed: ' . implode(', ', $allowed_statuses)]), JSON_UNESCAPED_UNICODE);
				break;
			}
			$updates[] = "post_status = ?";
			$update_fields[] = $new_status;
		}

		// Update post content if provided
		if (isset($input['content'])) {
			$new_content = $input['content'];
			$updates[] = "post_content = ?";
			$update_fields[] = $new_content;
		}

		if (empty($updates)) {
			$wp->close();
			http_response_code(400);
			echo json_encode(utf8ize(['error' => 'No updates provided']), JSON_UNESCAPED_UNICODE);
			break;
		}

		// Build the update query
		$update_sql = "UPDATE $WP_POSTS SET " . implode(', ', $updates) . ", post_modified = NOW(), post_modified_gmt = UTC_TIMESTAMP() WHERE ID = ?";
		$update_fields[] = $post_id;

		$update_stmt = $wp->prepare($update_sql);

		// Bind parameters dynamically
		$types = str_repeat('s', count($update_fields) - 1) . 'i'; // All strings except last (ID)
		$update_stmt->bind_param($types, ...$update_fields);

		if ($update_stmt->execute()) {
			$wp->close();
			$response = ['success' => true, 'post_id' => $post_id];
			if (isset($input['status'])) {
				$response['old_status'] = $current_status;
				$response['new_status'] = $input['status'];
			}
			if (isset($input['content'])) {
				$response['content_updated'] = true;
			}
			echo json_encode(utf8ize($response), JSON_UNESCAPED_UNICODE);
		} else {
			$wp->close();
			http_response_code(500);
			echo json_encode(utf8ize(['error' => 'Failed to update post']), JSON_UNESCAPED_UNICODE);
		}
		$update_stmt->close();
		break;

    // GET /posts/search
    case $method === 'GET' && preg_match('#^/posts/search$#', $uri):
		require_auth();
		global $WP_DB, $WP_POSTS;
		$q = isset($query['q']) ? trim($query['q']) : '';
		$limit = isset($query['limit']) && is_numeric($query['limit']) ? (int)$query['limit'] : 10;
		if ($q === '') {
			http_response_code(400);
			echo json_encode(['error' => 'Missing search query (q parameter)']);
			break;
		}
		$wp = get_wp_connection();
		// Search in post_title, post_content, post_excerpt (case-insensitive)
		$like = '%' . $wp->real_escape_string($q) . '%';
		$sql = "SELECT ID, post_title, post_date, post_name, post_excerpt, post_content FROM $WP_POSTS WHERE post_type = 'post' AND post_status = 'publish' AND (post_title LIKE ? OR post_content LIKE ? OR post_excerpt LIKE ?) ORDER BY post_date DESC LIMIT ?";
		$stmt = $wp->prepare($sql);
		$stmt->bind_param('sssi', $like, $like, $like, $limit);
		$stmt->execute();
		$result = $stmt->get_result();
		$posts = [];
		if ($result) {
			while ($row = $result->fetch_assoc()) {
				$post = [
					'id' => (int)$row['ID'],
					'title' => $row['post_title'],
					'slug' => $row['post_name'],
					'date' => $row['post_date'],
					'excerpt' => $row['post_excerpt'],
					'content' => $row['post_content'],
				];
				$posts[] = sanitize_quotes($post);
			}
		}
		$stmt->close();
		$wp->close();
		echo json_encode(utf8ize(['results' => $posts]), JSON_UNESCAPED_UNICODE);
		break;

	// POST /upload_image
	case $method === 'POST' && preg_match('#^/upload_image/?$#', $uri):
		require_auth();
		global $WP_POSTS, $WP_POSTMETA, $WP_PREFIX;

		$input = json_decode(file_get_contents('php://input'), true);
		if (!isset($input['image_data'], $input['filename'])) {
			http_response_code(400);
			echo json_encode(['error' => 'image_data and filename are required']);
			break;
		}

		$image_data = base64_decode($input['image_data']);
		$filename = $input['filename'];
		$base_url = isset($input['base_url']) ? rtrim($input['base_url'], '/') : 'https://' . $_SERVER['HTTP_HOST']; // Default fallback without WordPress
		$post_parent = isset($input['post_id']) ? (int)$input['post_id'] : 0; // Optional post parent
		$generate_metadata = isset($input['generate_metadata']) ? (bool)$input['generate_metadata'] : false; // Optional metadata generation - default false to reduce resources
		
		// Determine MIME type from filename extension
		$extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
		switch ($extension) {
			case 'jpg':
			case 'jpeg':
				$mime_type = 'image/jpeg';
				break;
			case 'png':
				$mime_type = 'image/png';
				break;
			case 'gif':
				$mime_type = 'image/gif';
				break;
			case 'webp':
				$mime_type = 'image/webp';
				break;
			default:
				$mime_type = 'image/jpeg'; // Default fallback
				break;
		}
		
		// Debug: Log the received base_url
		error_logger("Received base_url: " . (isset($input['base_url']) ? $input['base_url'] : 'NOT SET (using default)'));
		error_logger("Using base_url for GUID: " . $base_url);
		error_logger("Detected MIME type: " . $mime_type);

		if (!$image_data) {
			http_response_code(400);
			echo json_encode(['error' => 'Invalid base64 image data']);
			break;
		}

		// Debug: Check image data
		error_logger("Image data length: " . strlen($image_data));
		error_logger("First 100 bytes of decoded data: " . substr($image_data, 0, 100));

		// Create uploads directory if it doesn't exist
		$upload_dir = $_SERVER['DOCUMENT_ROOT'] . '/wp-content/uploads/' . date('Y') . '/' . date('m') . '/';
		error_logger("Upload directory: " . $upload_dir);
		if (!is_dir($upload_dir)) {
			if (!mkdir($upload_dir, 0755, true)) {
				http_response_code(500);
				echo json_encode(['error' => 'Failed to create upload directory']);
				break;
			}
		}

		// Generate unique filename
		$unique_filename = time() . '_' . $filename;
		$file_path = $upload_dir . $unique_filename;
		error_logger("File path: " . $file_path);

		// Save the image file
		if (file_put_contents($file_path, $image_data) === false) {
			error_logger("Failed to write file: " . $file_path);
			http_response_code(500);
			echo json_encode(['error' => 'Failed to save image file']);
			break;
		}

		// Check if file was actually created
		if (!file_exists($file_path)) {
			error_logger("File does not exist after creation: " . $file_path);
			http_response_code(500);
			echo json_encode(['error' => 'File was not created']);
			break;
		}

		$file_size = filesize($file_path);
		error_logger("File size: " . $file_size);

		// Note: Images are served from site URL
		error_logger("Checking if metadata generation is needed: " . ($generate_metadata ? 'yes' : 'no'));

		// Load WordPress (optional for metadata generation)
		$wordpress_loaded = false;
		if ($generate_metadata) {
			$wp_load_path = $_SERVER['DOCUMENT_ROOT'] . '/wp-load.php';
			if (file_exists($wp_load_path)) {
				try {
					require_once $wp_load_path;
					require_once ABSPATH . 'wp-admin/includes/image.php';
					global $wpdb;
					$wordpress_loaded = true;
					error_logger("WordPress loaded successfully for metadata generation");
				} catch (Exception $e) {
					error_logger("Failed to load WordPress: " . $e->getMessage());
				}
			} else {
				error_logger("wp-load.php not found at: " . $wp_load_path . " - proceeding without WordPress metadata generation");
			}
		} else {
			error_logger("Skipping WordPress load - metadata generation disabled");
		}

		$now = date('Y-m-d H:i:s');

		// Use direct database connection instead of requiring WordPress
		$wp_conn = get_wp_connection();
		if (!$wp_conn) {
			error_logger("Failed to connect to WordPress database");
			http_response_code(500);
			echo json_encode(['error' => 'Database connection failed']);
			break;
		}

		// Correct table prefix using config
		global $WP_PREFIX;
		$WP_POSTS = $WP_PREFIX . 'posts';
		$WP_POSTMETA = $WP_PREFIX . 'postmeta';

		// Test if the posts table exists
		$test_sql = "SHOW TABLES LIKE '{$WP_POSTS}'";
		$test_result = $wp_conn->query($test_sql);
		if (!$test_result || $test_result->num_rows === 0) {
			error_logger("Posts table '{$WP_POSTS}' does not exist");
			$wp_conn->close();
			http_response_code(500);
			echo json_encode(['error' => 'Posts table not found']);
			break;
		}
		$test_result->free();

		// Test if the postmeta table exists
		$test_sql = "SHOW TABLES LIKE '{$WP_POSTMETA}'";
		$test_result = $wp_conn->query($test_sql);
		if (!$test_result || $test_result->num_rows === 0) {
			error_logger("Postmeta table '{$WP_POSTMETA}' does not exist");
			$wp_conn->close();
			http_response_code(500);
			echo json_encode(['error' => 'Postmeta table not found']);
			break;
		}
		$test_result->free();

		// Prepare variables for binding
		$post_author = 1;
		$current_date = date('Y-m-d H:i:s');
		$guid_url = $base_url . '/wp-content/uploads/' . date('Y') . '/' . date('m') . '/' . $unique_filename;

		// Insert attachment post using direct SQL
		$sql = "INSERT INTO {$WP_POSTS} (post_author, post_date, post_date_gmt, post_content, post_title, post_excerpt, post_status, comment_status, ping_status, post_name, guid, post_type, post_mime_type, post_modified, post_modified_gmt, post_parent) VALUES (?, ?, ?, '', ?, '', 'inherit', 'open', 'closed', ?, ?, 'attachment', ?, ?, ?, ?)";
		$stmt = $wp_conn->prepare($sql);
		if (!$stmt) {
			error_logger("Failed to prepare insert statement: " . $wp_conn->error);
			$wp_conn->close();
			http_response_code(500);
			echo json_encode(['error' => 'SQL prepare failed']);
			break;
		}

		// Prepare variables for binding
		$post_author_val = 1;
		$current_date_val = date('Y-m-d H:i:s');
		$guid_url_val = $base_url . '/wp-content/uploads/' . date('Y') . '/' . date('m') . '/' . $unique_filename;
		$mime_type_val = $mime_type;
		$post_parent_val = $post_parent;

		$stmt->bind_param('issssssssi', 
			$post_author_val,
			$current_date_val,
			$current_date_val,
			$filename,
			$unique_filename,
			$guid_url_val,
			$mime_type_val,
			$current_date_val,
			$current_date_val,
			$post_parent_val
		);

		if (!$stmt->execute()) {
			$error_msg = "Failed to insert attachment post: " . $stmt->error . " | SQL: " . $sql;
			error_logger($error_msg);
			$stmt->close();
			$wp_conn->close();
			http_response_code(500);
			echo json_encode(['error' => 'Failed to create attachment post']);
			break;
		}

		$attachment_id = $stmt->insert_id;
		$stmt->close();
		error_logger("Attachment post inserted with ID: " . $attachment_id);

		// Generate proper WordPress attachment metadata (optional)
		if ($generate_metadata && $wordpress_loaded) {
			if (function_exists('wp_generate_attachment_metadata')) {
				try {
					$metadata = wp_generate_attachment_metadata($attachment_id, $file_path);
					if ($metadata === false) {
						error_logger("Failed to generate attachment metadata");
					} else {
						wp_update_attachment_metadata($attachment_id, $metadata);
						error_logger("Attachment metadata generated and updated");
					}
				} catch (Exception $e) {
					error_logger("Error during metadata generation: " . $e->getMessage());
				}
			} else {
				error_logger("wp_generate_attachment_metadata function not available, skipping metadata generation");
			}
		} else {
			if (!$generate_metadata) {
				error_logger("Skipping metadata generation - disabled by client");
			} else {
				error_logger("Skipping metadata generation - WordPress not available");
			}
		}

		// Add postmeta for attachment file path
		$meta_sql = "INSERT INTO {$WP_POSTMETA} (post_id, meta_key, meta_value) VALUES (?, '_wp_attached_file', ?)";
		$meta_stmt = $wp_conn->prepare($meta_sql);
		if ($meta_stmt) {
			$meta_value = date('Y') . '/' . date('m') . '/' . $unique_filename;
			$meta_stmt->bind_param('is', $attachment_id, $meta_value);
			$meta_stmt->execute();
			$meta_stmt->close();
		}

		$wp_conn->close();

		// Return success response
		$guid = $guid_url_val;
		echo json_encode(['attachment_id' => $attachment_id, 'url' => $guid]);
		break;

	// POST /upload_file
	case $method === 'POST' && preg_match('#^/upload_file/?$#', $uri):
		require_auth();
		try {
			$input = json_decode(file_get_contents('php://input'), true);
			if (!isset($input['image_data'], $input['filename'])) {
				http_response_code(400);
				echo json_encode(create_error_response('VALIDATION_ERROR', 'image_data and filename are required'));
				break;
			}

			$image_data = base64_decode($input['image_data']);
			$filename = $input['filename'];

			if (!$image_data) {
				http_response_code(400);
				echo json_encode(create_error_response('VALIDATION_ERROR', 'Invalid base64 image data', null, ['filename' => $filename]));
				break;
			}

			error_logger("Starting file upload", 'INFO', ['filename' => $filename, 'data_size' => strlen($input['image_data'])]);

			// Get upload directory
			$upload_year = date('Y');
			$upload_month = date('m');
			$upload_dir = [
				'basedir' => $_SERVER['DOCUMENT_ROOT'] . '/wp-content/uploads/' . $upload_year . '/' . $upload_month,
				'baseurl' => 'https://' . $_SERVER['HTTP_HOST'] . '/wp-content/uploads/' . $upload_year . '/' . $upload_month
			];
			$upload_path = $upload_dir['basedir'];
			if (!is_dir($upload_path)) {
				if (!mkdir($upload_path, 0755, true)) {
					http_response_code(500);
					echo json_encode(['error' => 'Failed to create upload directory']);
					break;
				}
			}
			// Generate unique filename
			$unique_filename = time() . '_' . $filename;
			$file_path = $upload_path . '/' . $unique_filename;
			// Save file
			if (file_put_contents($file_path, $image_data) === false) {
				http_response_code(500);
				echo json_encode(['error' => 'Failed to save image file']);
				break;
			}
			if (!file_exists($file_path)) {
				http_response_code(500);
				echo json_encode(['error' => 'File saved but not found', 'path' => $file_path]);
				break;
			}
			$url = $upload_dir['baseurl'] . '/' . $unique_filename;
			echo json_encode(['file_path' => $file_path, 'url' => $url, 'size' => filesize($file_path)]);
			break;
		} catch (Exception $e) {
			error_logger("Upload file error: " . $e->getMessage(), 'ERROR', ['filename' => $filename ?? 'unknown']);
			http_response_code(500);
			echo json_encode(create_error_response('UPLOAD_ERROR', 'Failed to upload file', $e->getMessage()));
			break;
		}

	// POST /upload_file_multipart
	case $method === 'POST' && preg_match('#^/upload_file_multipart/?$#', $uri):
		require_auth();
		try {
			// Debug logging for multipart upload
			error_logger("Multipart upload request received", 'INFO', [
				'content_type' => $_SERVER['CONTENT_TYPE'] ?? 'not set',
				'request_method' => $_SERVER['REQUEST_METHOD'],
				'files_count' => count($_FILES),
				'post_count' => count($_POST),
				'raw_post_data_size' => strlen(file_get_contents('php://input'))
			]);

			// Log all $_FILES contents
			error_logger("FILES array contents", 'DEBUG', ['files' => $_FILES]);

			// Log all $_POST contents
			error_logger("POST array contents", 'DEBUG', ['post' => $_POST]);

			// Handle multipart upload
			if (!isset($_FILES['image_file'])) {
				error_logger("image_file not found in FILES array", 'ERROR', [
					'available_files' => array_keys($_FILES),
					'files_details' => $_FILES
				]);
				http_response_code(400);
				echo json_encode(['error' => 'image_file is required']);
				break;
			}

			$file = $_FILES['image_file'];
			$filename = isset($_POST['filename']) ? $_POST['filename'] : $file['name'];

			error_logger("Processing uploaded file", 'INFO', [
				'filename' => $filename,
				'file_error' => $file['error'],
				'file_size' => $file['size'] ?? 'unknown',
				'file_tmp_name' => $file['tmp_name'] ?? 'unknown',
				'file_type' => $file['type'] ?? 'unknown'
			]);

			// Validate file
			if ($file['error'] !== UPLOAD_ERR_OK) {
				error_logger("File upload error", 'ERROR', [
					'error_code' => $file['error'],
					'error_description' => [
						UPLOAD_ERR_INI_SIZE => 'The uploaded file exceeds the upload_max_filesize directive',
						UPLOAD_ERR_FORM_SIZE => 'The uploaded file exceeds the MAX_FILE_SIZE directive',
						UPLOAD_ERR_PARTIAL => 'The uploaded file was only partially uploaded',
						UPLOAD_ERR_NO_FILE => 'No file was uploaded',
						UPLOAD_ERR_NO_TMP_DIR => 'Missing a temporary folder',
						UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
						UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the file upload'
					][$file['error']] ?? 'Unknown error'
				]);
				http_response_code(400);
				echo json_encode(['error' => 'Upload failed: ' . $file['error']]);
				break;
			}

			// Generate upload path
			$upload_year = date('Y');
			$upload_month = date('m');
			$upload_dir = $_SERVER['DOCUMENT_ROOT'] . '/wp-content/uploads/' . $upload_year . '/' . $upload_month;

			if (!is_dir($upload_dir)) {
				mkdir($upload_dir, 0755, true);
			}

			$unique_filename = time() . '_' . $filename;
			$file_path = $upload_dir . '/' . $unique_filename;

			// Move uploaded file
			if (!move_uploaded_file($file['tmp_name'], $file_path)) {
				error_logger("Failed to move uploaded file", 'ERROR', [
					'tmp_name' => $file['tmp_name'],
					'destination' => $file_path
				]);
				http_response_code(500);
				echo json_encode(['error' => 'Failed to save uploaded file']);
				break;
			}

			$url = 'https://' . $_SERVER['HTTP_HOST'] . '/wp-content/uploads/' . $upload_year . '/' . $upload_month . '/' . $unique_filename;

			error_logger("File upload successful", 'INFO', [
				'file_path' => $file_path,
				'url' => $url,
				'size' => filesize($file_path)
			]);

			echo json_encode([
				'file_path' => $file_path,
				'url' => $url,
				'size' => filesize($file_path)
			]);
			break;

		} catch (Exception $e) {
			error_logger("Multipart upload error: " . $e->getMessage());
			http_response_code(500);
			echo json_encode(['error' => 'Upload failed']);
			break;
		}

	// POST /create_attachment
	case $method === 'POST' && preg_match('#^/create_attachment/?$#', $uri):
		require_auth();
		try {
			$input = json_decode(file_get_contents('php://input'), true);
			if (!isset($input['unique_filename'], $input['filename'])) {
				http_response_code(400);
				echo json_encode(create_error_response('VALIDATION_ERROR', 'unique_filename and filename are required'));
				break;
			}

			$unique_filename = $input['unique_filename'];
			$filename = $input['filename'];
			$base_url = isset($input['base_url']) ? rtrim($input['base_url'], '/') : 'https://' . $_SERVER['HTTP_HOST'];
			$post_parent = isset($input['post_id']) ? (int)$input['post_id'] : 0;

			error_logger("Creating attachment post", 'INFO', [
				'unique_filename' => $unique_filename,
				'filename' => $filename,
				'post_id' => $post_parent
			]);
		// Determine MIME type
		$mime_type = 'image/jpeg'; // Default
		$extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
		switch ($extension) {
			case 'jpg':
			case 'jpeg':
				$mime_type = 'image/jpeg';
				break;
			case 'png':
				$mime_type = 'image/png';
				break;
			case 'gif':
				$mime_type = 'image/gif';
				break;
			case 'webp':
				$mime_type = 'image/webp';
				break;
			default:
				$mime_type = 'image/jpeg';
				break;
		}
		$wp_conn = get_wp_connection();
		if (!$wp_conn) {
			http_response_code(500);
			echo json_encode(['error' => 'Database connection failed']);
			break;
		}
		// Test tables
		$test_sql = "SHOW TABLES LIKE '{$WP_POSTS}'";
		$test_result = $wp_conn->query($test_sql);
		if (!$test_result || $test_result->num_rows === 0) {
			$wp_conn->close();
			http_response_code(500);
			echo json_encode(['error' => 'Posts table not found']);
			break;
		}
		$test_result->free();
		// Insert attachment post
		$current_date = date('Y-m-d H:i:s');
		$upload_year = date('Y');
		$upload_month = date('m');
		$upload_dir = [
			'basedir' => $_SERVER['DOCUMENT_ROOT'] . '/wp-content/uploads/' . $upload_year . '/' . $upload_month,
			'baseurl' => 'https://' . $_SERVER['HTTP_HOST'] . '/wp-content/uploads/' . $upload_year . '/' . $upload_month
		];
		$guid_url = $upload_dir['baseurl'] . '/' . $unique_filename;
		$empty = '';
		$sql = "INSERT INTO {$WP_POSTS} (post_author, post_date, post_date_gmt, post_content, post_title, post_excerpt, post_status, post_name, to_ping, pinged, post_modified, post_modified_gmt, post_content_filtered, post_parent, menu_order, post_type, post_mime_type, comment_count, guid) VALUES (1, ?, ?, '', ?, '', 'inherit', '', '', '', ?, ?, '', ?, 0, 'attachment', ?, 0, ?)";
		$stmt = $wp_conn->prepare($sql);
		if (!$stmt) {
			$wp_conn->close();
			http_response_code(500);
			echo json_encode(['error' => 'SQL prepare failed']);
			break;
		}
		$stmt->bind_param('ssssssss', $current_date, $current_date, $filename, $current_date, $current_date, $post_parent, $mime_type, $guid_url);
		if (!$stmt->execute()) {
			$stmt->close();
			$wp_conn->close();
			http_response_code(500);
			echo json_encode(['error' => 'Failed to create attachment post']);
			break;
		}
		$attachment_id = $stmt->insert_id;
		$stmt->close();
		$wp_conn->close();
		echo json_encode(['attachment_id' => $attachment_id, 'url' => $guid_url]);
		break;
		} catch (Exception $e) {
			error_logger("Create attachment error: " . $e->getMessage(), 'ERROR', [
				'unique_filename' => $unique_filename ?? 'unknown',
				'filename' => $filename ?? 'unknown'
			]);
			http_response_code(500);
			echo json_encode(create_error_response('ATTACHMENT_ERROR', 'Failed to create attachment', $e->getMessage()));
			break;
		}

	// POST /generate_metadata
	case $method === 'POST' && preg_match('#^/generate_metadata/?$#', $uri):
		require_auth();
		try {
			$input = json_decode(file_get_contents('php://input'), true);
			if (!isset($input['attachment_id'], $input['unique_filename'])) {
				http_response_code(400);
				echo json_encode(create_error_response('VALIDATION_ERROR', 'attachment_id and unique_filename are required'));
				break;
			}

			$attachment_id = (int)$input['attachment_id'];
			$unique_filename = $input['unique_filename'];

			error_logger("Starting metadata generation", 'INFO', [
				'attachment_id' => $attachment_id,
				'unique_filename' => $unique_filename
			]);

			// Load WordPress
			$wp_load_path = $_SERVER['DOCUMENT_ROOT'] . '/wp-load.php';
			if (!file_exists($wp_load_path)) {
				http_response_code(500);
				echo json_encode(create_error_response('WORDPRESS_ERROR', 'WordPress not available', null, ['wp_load_path' => $wp_load_path]));
				break;
			}
		
			// Set a reasonable timeout for metadata generation (45 seconds to match client timeout)
			set_time_limit(45);
			require_once $wp_load_path;
			error_logger("WordPress loaded successfully");

			require_once ABSPATH . 'wp-admin/includes/image.php';
			error_logger("Image functions loaded");

			if (!function_exists('wp_generate_attachment_metadata')) {
				http_response_code(500);
				echo json_encode(create_error_response('WORDPRESS_ERROR', 'WordPress image functions not available'));
				break;
			}

			// Reconstruct file_path
			$upload_year = date('Y');
			$upload_month = date('m');
			$file_path = $_SERVER['DOCUMENT_ROOT'] . '/wp-content/uploads/' . $upload_year . '/' . $upload_month . '/' . $unique_filename;

			error_logger("Generating metadata", 'INFO', [
				'attachment_id' => $attachment_id,
				'file_path' => $file_path
			]);

			$metadata = wp_generate_attachment_metadata($attachment_id, $file_path);
			if ($metadata === false) {
				error_logger("Metadata generation returned false", 'WARNING', ['attachment_id' => $attachment_id]);
			} else {
				wp_update_attachment_metadata($attachment_id, $metadata);
				error_logger("Metadata generated and updated successfully", 'INFO', [
					'attachment_id' => $attachment_id,
					'metadata_size' => strlen(json_encode($metadata))
				]);
			}

			echo json_encode(['success' => true, 'metadata' => $metadata]);
			break;

		} catch (Exception $e) {
			error_logger("Metadata generation error: " . $e->getMessage(), 'ERROR', [
				'attachment_id' => $attachment_id ?? 'unknown',
				'unique_filename' => $unique_filename ?? 'unknown'
			]);
			http_response_code(500);
			echo json_encode(create_error_response('METADATA_ERROR', 'Failed to generate metadata', $e->getMessage()));
			break;
		}

	// POST /add_postmeta
	case $method === 'POST' && preg_match('#^/add_postmeta/?$#', $uri):
		require_auth();
		global $WP_POSTMETA, $WP_PREFIX;
		$input = json_decode(file_get_contents('php://input'), true);
		if (!isset($input['attachment_id'], $input['unique_filename'])) {
			http_response_code(400);
			echo json_encode(['error' => 'attachment_id and unique_filename are required']);
			break;
		}
		$attachment_id = (int)$input['attachment_id'];
		$unique_filename = $input['unique_filename'];
		$wp_conn = get_wp_connection();
		if (!$wp_conn) {
			http_response_code(500);
			echo json_encode(['error' => 'Database connection failed']);
			break;
		}
		// Correct table prefix
		global $WP_PREFIX;
		$WP_POSTMETA = $WP_PREFIX . 'postmeta';
		// Test postmeta table
		$test_sql = "SHOW TABLES LIKE '{$WP_POSTMETA}'";
		$test_result = $wp_conn->query($test_sql);
		if (!$test_result || $test_result->num_rows === 0) {
			$wp_conn->close();
			http_response_code(500);
			echo json_encode(['error' => 'Postmeta table not found']);
			break;
		}
		$test_result->free();
		// Add postmeta
		$meta_value = date('Y') . '/' . date('m') . '/' . $unique_filename;
		$meta_sql = "INSERT INTO {$WP_POSTMETA} (post_id, meta_key, meta_value) VALUES (?, '_wp_attached_file', ?)";
		$meta_stmt = $wp_conn->prepare($meta_sql);
		if ($meta_stmt) {
			$meta_stmt->bind_param('is', $attachment_id, $meta_value);
			$meta_stmt->execute();
			$meta_stmt->close();
		}
		$wp_conn->close();
		echo json_encode(['success' => true]);
		break;

	// POST /posts/{post_id}
	case $method === 'POST' && preg_match('#^/posts/(\d+)/?$#', $uri, $matches):
		require_auth();
		$post_id = (int)$matches[1];
		global $WP_POSTMETA, $WP_PREFIX;

		error_logger("Setting featured image for post $post_id");

		$input = json_decode(file_get_contents('php://input'), true);
		if (!isset($input['featured_media'])) {
			http_response_code(400);
			echo json_encode(['error' => 'featured_media is required']);
			break;
		}

		$attachment_id = (int)$input['featured_media'];
		error_logger("Attachment ID: $attachment_id");

		$wp = get_wp_connection();
		if (!$wp) {
			error_logger("Failed to connect to WP DB");
			http_response_code(500);
			echo json_encode(['error' => 'Database connection failed']);
			break;
		}

		// First, remove any existing featured image
		$delete_sql = "DELETE FROM {$WP_PREFIX}postmeta WHERE post_id = ? AND meta_key = '_thumbnail_id'";
		error_logger("Delete SQL: $delete_sql");
		$delete_stmt = $wp->prepare($delete_sql);
		if (!$delete_stmt) {
			error_logger("Delete prepare failed: " . $wp->error);
			http_response_code(500);
			echo json_encode(['error' => 'SQL prepare failed']);
			$wp->close();
			break;
		}
		$delete_stmt->bind_param('i', $post_id);
		$delete_result = $delete_stmt->execute();
		error_logger("Delete result: " . ($delete_result ? 'success' : 'failed'));
		$delete_stmt->close();

		// Set the new featured image
		$meta_key = '_thumbnail_id';
		error_logger("About to set featured image: post_id=$post_id, attachment_id=$attachment_id");
		$insert_sql = "REPLACE INTO {$WP_PREFIX}postmeta (post_id, meta_key, meta_value) VALUES (?, ?, ?)";
		$insert_stmt = $wp->prepare($insert_sql);
		if (!$insert_stmt) {
			error_logger("Failed to prepare REPLACE statement: " . $wp->error);
			http_response_code(500);
			echo json_encode(['error' => 'SQL prepare failed']);
			$wp->close();
			break;
		}
		$insert_stmt->bind_param('isi', $post_id, $meta_key, $attachment_id);
		$success = $insert_stmt->execute();
		error_logger("REPLACE execute result: " . ($success ? 'success' : 'failed'));
		$insert_stmt->close();

		// Update attachment context and alt text
		if ($success) {
			// Also update the main post's modified date
			$update_post_sql = "UPDATE $WP_POSTS SET post_modified = ?, post_modified_gmt = ? WHERE ID = ?";
			$update_post_stmt = $wp->prepare($update_post_sql);
			$update_post_stmt->bind_param('ssi', $now, $now, $post_id);
			$update_post_stmt->execute();
			$update_post_stmt->close();

			// Set attachment context and alt text
			$context_key = '_wp_attachment_context';
			$context_value = 'featured-image';
			$context_sql = "INSERT INTO {$WP_PREFIX}postmeta (post_id, meta_key, meta_value) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE meta_value = VALUES(meta_value)";
			$context_stmt = $wp->prepare($context_sql);
			$context_stmt->bind_param('iss', $attachment_id, $context_key, $context_value);
			$context_stmt->execute();
			$context_stmt->close();

			// Set alt text if provided, otherwise use a default
			$alt_key = '_wp_attachment_image_alt';
			$alt_text = isset($input['alt_text']) ? $input['alt_text'] : 'Featured image';
			$alt_sql = "INSERT INTO {$WP_PREFIX}postmeta (post_id, meta_key, meta_value) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE meta_value = VALUES(meta_value)";
			$alt_stmt = $wp->prepare($alt_sql);
			$alt_stmt->bind_param('iss', $attachment_id, $alt_key, $alt_text);
			$alt_stmt->execute();
			$alt_stmt->close();
		}

		$wp->close();

		if ($success) {
			echo json_encode(['success' => true, 'post_id' => $post_id, 'featured_media' => $attachment_id]);
		} else {
			http_response_code(500);
			echo json_encode(['error' => 'Failed to set featured image']);
		}
		break;

	// GET /categories
	case $method === 'GET' && preg_match('#^/categories/?$#', $uri):
		require_auth();
		$categories = get_categories(['hide_empty' => false]);
		echo json_encode($categories);
		break;

	// POST /init_affiliate_db
	case $method === 'POST' && preg_match('#^/init_affiliate_db/?$#', $uri):
		error_logger("Endpoint /init_affiliate_db matched, method: $method, uri: $uri");
		require_auth();
		error_logger("Authentication passed");
		try {
			$input = json_decode(file_get_contents('php://input'), true);
			error_logger("Starting affiliate database initialization");
						
			// Connect to the WordPress database for affiliate operations
			$conn = get_wp_connection();
			error_logger("Database connection established successfully");
			error_logger("WP_PREFIX: $WP_PREFIX");
			
			// Force drop if requested
			if (isset($input['force']) && $input['force']) {
				$drop_sql = "DROP TABLE IF EXISTS " . $WP_PREFIX . "ap_affiliate_links";
				$conn->query($drop_sql);
				error_logger("Dropped existing affiliate_links table for force migration");
			}
			
			// Check if affiliate_links table already exists in WordPress DB
			$check_sql = "SELECT 1 FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = '" . $WP_PREFIX . "ap_affiliate_links' LIMIT 1";
			$result = $conn->query($check_sql);
			if ($result && $result->num_rows > 0) {
				$conn->close();
				error_logger("Affiliate database already initialized in WordPress DB");
				echo json_encode(['success' => true, 'message' => 'Affiliate database already initialized']);
				break;
			}
			
			// Check if table exists in old database and migrate data
			$old_conn = get_mysql_connection();
			$old_check_sql = "SELECT 1 FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = 'affiliate_links' LIMIT 1";
			$old_result = $old_conn->query($old_check_sql);
			$migrated = false;
			
			if ($old_result && $old_result->num_rows > 0) {
				error_logger("Found existing affiliate_links table in old database, migrating data...");
				
				// Create table in WordPress DB first
				$create_table_sql = "
				CREATE TABLE " . $WP_PREFIX . "ap_affiliate_links (
					id INT AUTO_INCREMENT PRIMARY KEY,
					post_id INT NOT NULL,
					affiliate_program VARCHAR(255) NOT NULL,
					merchant_domain VARCHAR(255) NOT NULL,
					product_name VARCHAR(500) NOT NULL,
					product_url TEXT NOT NULL,
					affiliate_url TEXT NOT NULL,
					product_description TEXT,
					is_active TINYINT(1) DEFAULT 1,
					created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
					updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
					INDEX idx_post_id (post_id),
					INDEX idx_affiliate_program (affiliate_program),
					INDEX idx_merchant_domain (merchant_domain),
					INDEX idx_is_active (is_active)
				) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
				";
				
				if ($conn->query($create_table_sql) === TRUE) {
					// Migrate data from old database by reading and inserting
					$select_sql = "SELECT id, post_id, affiliate_program, merchant_domain, product_name, product_url, affiliate_url, product_description, is_active, created_at, updated_at FROM affiliate_links";
					$old_stmt = $old_conn->prepare($select_sql);
					$old_stmt->execute();
					$old_result = $old_stmt->get_result();
					
					$migrated_count = 0;
					$insert_sql = "INSERT INTO " . $WP_PREFIX . "ap_affiliate_links (id, post_id, affiliate_program, merchant_domain, product_name, product_url, affiliate_url, product_description, is_active, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
					$insert_stmt = $conn->prepare($insert_sql);
					
					while ($row = $old_result->fetch_assoc()) {
						$insert_stmt->bind_param('iisssssisss', 
							$row['id'], $row['post_id'], $row['affiliate_program'], $row['merchant_domain'], 
							$row['product_name'], $row['product_url'], $row['affiliate_url'], 
							$row['product_description'], $row['is_active'], $row['created_at'], $row['updated_at']
						);
						$insert_stmt->execute();
						$migrated_count++;
					}
					
					$insert_stmt->close();
					$old_stmt->close();
					
					error_logger("Successfully migrated $migrated_count affiliate links from old database");
					$migrated = true;
				} else {
					error_logger("Failed to create table in WordPress DB: " . $conn->error);
				}
				
				$old_conn->close();
			} else {
				// Create table normally if no old data exists
				$create_table_sql = "
				CREATE TABLE " . $WP_PREFIX . "ap_affiliate_links (
					id INT AUTO_INCREMENT PRIMARY KEY,
					post_id INT NOT NULL,
					affiliate_program VARCHAR(255) NOT NULL,
					merchant_domain VARCHAR(255) NOT NULL,
					product_name VARCHAR(500) NOT NULL,
					product_url TEXT NOT NULL,
					affiliate_url TEXT NOT NULL,
					product_description TEXT,
					is_active TINYINT(1) DEFAULT 1,
					created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
					updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
					INDEX idx_post_id (post_id),
					INDEX idx_affiliate_program (affiliate_program),
					INDEX idx_merchant_domain (merchant_domain),
					INDEX idx_is_active (is_active)
				) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
				";
				
				error_logger("Executing table creation SQL");
				if ($conn->query($create_table_sql) === TRUE) {
					error_logger("Affiliate database initialized successfully");
				} else {
					$error_msg = "Failed to create affiliate_links table: " . $conn->error;
					error_logger($error_msg);
					$conn->close();
					http_response_code(500);
					echo json_encode(['error' => $error_msg]);
					break;
				}
			}
			
			$conn->close();
			if ($migrated) {
				echo json_encode(['success' => true, 'message' => 'Affiliate database migrated successfully']);
			} else {
				echo json_encode(['success' => true, 'message' => 'Affiliate database initialized successfully']);
			}
			break;
			
		} catch (Exception $e) {
			$error_msg = "Exception initializing affiliate database: " . $e->getMessage();
			error_logger($error_msg);
			http_response_code(500);
			echo json_encode(['error' => $error_msg]);
			break;
		}

	// POST /migrate_affiliate_data
	case $method === 'POST' && preg_match('#^/migrate_affiliate_data/?$#', $uri):
		require_auth();
		try {
			error_logger("Starting affiliate data migration");
			
			// Connect to both databases
			$wp_conn = get_wp_connection();
			$old_conn = get_mysql_connection();
			
			// Check if old table exists and has data
			$old_check_sql = "SHOW TABLES LIKE 'affiliate_links'";
			$old_result = $old_conn->query($old_check_sql);
			
			if (!$old_result || $old_result->num_rows == 0) {
				$wp_conn->close();
				$old_conn->close();
				echo json_encode(['success' => false, 'message' => 'No affiliate data found in old database']);
				break;
			}
			
			// Check if new table exists, create if not
			$wp_check_sql = "SELECT 1 FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = '" . $WP_PREFIX . "ap_affiliate_links' LIMIT 1";
			$wp_result = $wp_conn->query($wp_check_sql);
			
			if (!$wp_result || $wp_result->num_rows == 0) {
				// Create table in WordPress DB
				$create_table_sql = "
				CREATE TABLE " . $WP_PREFIX . "ap_affiliate_links (
					id INT AUTO_INCREMENT PRIMARY KEY,
					post_id INT NOT NULL,
					affiliate_program VARCHAR(255) NOT NULL,
					merchant_domain VARCHAR(255) NOT NULL,
					product_name VARCHAR(500) NOT NULL,
					product_url TEXT NOT NULL,
					affiliate_url TEXT NOT NULL,
					product_description TEXT,
					is_active TINYINT(1) DEFAULT 1,
					created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
					updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
					INDEX idx_post_id (post_id),
					INDEX idx_affiliate_program (affiliate_program),
					INDEX idx_merchant_domain (merchant_domain),
					INDEX idx_is_active (is_active)
				) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
				";
				
				if ($wp_conn->query($create_table_sql) !== TRUE) {
					$wp_conn->close();
					$old_conn->close();
					http_response_code(500);
					echo json_encode(['error' => 'Failed to create affiliate_links table in WordPress DB']);
					break;
				}
			}
			
			// Migrate data from old database
			$select_sql = "SELECT id, post_id, affiliate_program, merchant_domain, product_name, product_url, affiliate_url, COALESCE(product_description, '') as product_description, is_active, created_at, updated_at FROM affiliate_links";
			$old_stmt = $old_conn->prepare($select_sql);
			$old_stmt->execute();
			$old_result = $old_stmt->get_result();
			
			$migrated_count = 0;
			
			while ($row = $old_result->fetch_assoc()) {
				// Use direct INSERT instead of prepared statement to avoid bind_param issues
				$id = (int)$row['id'];
				$post_id = (int)$row['post_id'];
				$affiliate_program = $wp_conn->real_escape_string($row['affiliate_program']);
				$merchant_domain = $wp_conn->real_escape_string($row['merchant_domain']);
				$product_name = $wp_conn->real_escape_string($row['product_name']);
				$product_url = $wp_conn->real_escape_string($row['product_url']);
				$affiliate_url = $wp_conn->real_escape_string($row['affiliate_url']);
				$product_description = $wp_conn->real_escape_string($row['product_description']);
				$is_active = (int)$row['is_active'];
				$created_at = $row['created_at'];
				$updated_at = $row['updated_at'];
				
				$insert_sql = "INSERT INTO " . $WP_PREFIX . "ap_affiliate_links (id, post_id, affiliate_program, merchant_domain, product_name, product_url, affiliate_url, product_description, is_active, created_at, updated_at) 
					VALUES ($id, $post_id, '$affiliate_program', '$merchant_domain', '$product_name', '$product_url', '$affiliate_url', '$product_description', $is_active, '$created_at', '$updated_at') 
					ON DUPLICATE KEY UPDATE post_id=$post_id, is_active=$is_active, updated_at='$updated_at'";
				
				if ($wp_conn->query($insert_sql) === TRUE) {
					$migrated_count++;
				} else {
					error_logger("Failed to insert affiliate link $id: " . $wp_conn->error);
				}
			}
			$old_stmt->close();
			$wp_conn->close();
			$old_conn->close();
			
			error_logger("Successfully migrated $migrated_count affiliate links");
			echo json_encode(['success' => true, 'message' => "Migrated $migrated_count affiliate links from old database"]);
			
		} catch (Exception $e) {
			$error_msg = "Exception migrating affiliate data: " . $e->getMessage();
			error_logger($error_msg);
			http_response_code(500);
			echo json_encode(['error' => $error_msg]);
			break;
		}

	// POST /store_affiliate_links
	case $method === 'POST' && preg_match('#^/store_affiliate_links/?$#', $uri):
		require_auth();
		try {
			error_logger("Storing affiliate links");
			$input = json_decode(file_get_contents('php://input'), true);
			
			if (!isset($input['post_id'], $input['links']) || !is_array($input['links'])) {
				http_response_code(400);
				echo json_encode(['error' => 'post_id and links array required']);
				break;
			}
			
			$post_id = (int)$input['post_id'];
			$links = $input['links'];
			
			$conn = get_wp_connection();
			$table_name = $WP_PREFIX . 'ap_affiliate_links';
			$stored_ids = [];
			
			foreach ($links as $link_data) {
				$sql = "INSERT INTO $table_name 
						(post_id, affiliate_program, merchant_domain, product_name, 
						 product_url, affiliate_url, product_description, is_active) 
						VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
				
				$stmt = $conn->prepare($sql);
				
				// Extract values to variables (required for bind_param by reference)
				$affiliate_program = $link_data['affiliate_program'] ?? '';
				$merchant_domain = $link_data['merchant_domain'] ?? '';
				$product_name = $link_data['product_name'] ?? '';
				$product_url = $link_data['product_url'] ?? '';
				$affiliate_url = $link_data['affiliate_url'] ?? '';
				$product_description = $link_data['product_description'] ?? '';
				$is_active = isset($link_data['is_active']) ? (int)$link_data['is_active'] : 1;
				
				$stmt->bind_param('issssssi', 
					$post_id,
					$affiliate_program,
					$merchant_domain,
					$product_name,
					$product_url,
					$affiliate_url,
					$product_description,
					$is_active
				);
				
				if ($stmt->execute()) {
					$stored_ids[] = $stmt->insert_id;
				}
				$stmt->close();
			}
			
			// Automatically add 'affiliate' tag to the post if it doesn't already have it
			if (!empty($stored_ids)) {
				$terms_table = $WP_PREFIX . 'terms';
				$term_taxonomy_table = $WP_PREFIX . 'term_taxonomy';
				$term_relationships_table = $WP_PREFIX . 'term_relationships';
				
				// Check if 'affiliate' term exists
				$term_sql = "SELECT term_id FROM $terms_table WHERE name = 'affiliate' AND slug = 'affiliate' LIMIT 1";
				$term_result = $conn->query($term_sql);
				
				if ($term_result && $term_result->num_rows > 0) {
					$term_row = $term_result->fetch_assoc();
					$term_id = $term_row['term_id'];
				} else {
					// Insert new term
					$insert_term_sql = "INSERT INTO $terms_table (name, slug) VALUES ('affiliate', 'affiliate')";
					if ($conn->query($insert_term_sql)) {
						$term_id = $conn->insert_id;
					} else {
						error_logger("Failed to insert affiliate term");
						$term_id = null;
					}
				}
				
				if ($term_id) {
					// Get or create term_taxonomy
					$taxonomy_sql = "SELECT term_taxonomy_id FROM $term_taxonomy_table WHERE term_id = $term_id AND taxonomy = 'post_tag' LIMIT 1";
					$taxonomy_result = $conn->query($taxonomy_sql);
					
					if ($taxonomy_result && $taxonomy_result->num_rows > 0) {
						$taxonomy_row = $taxonomy_result->fetch_assoc();
						$term_taxonomy_id = $taxonomy_row['term_taxonomy_id'];
					} else {
						$insert_taxonomy_sql = "INSERT INTO $term_taxonomy_table (term_id, taxonomy, description, parent, count) VALUES ($term_id, 'post_tag', '', 0, 0)";
						if ($conn->query($insert_taxonomy_sql)) {
							$term_taxonomy_id = $conn->insert_id;
						} else {
							error_logger("Failed to insert term taxonomy");
							$term_taxonomy_id = null;
						}
					}
					
					if ($term_taxonomy_id) {
						// Check if relationship exists
						$rel_sql = "SELECT 1 FROM $term_relationships_table WHERE object_id = $post_id AND term_taxonomy_id = $term_taxonomy_id LIMIT 1";
						$rel_result = $conn->query($rel_sql);
						
						if (!$rel_result || $rel_result->num_rows == 0) {
							// Insert relationship
							$insert_rel_sql = "INSERT INTO $term_relationships_table (object_id, term_taxonomy_id, term_order) VALUES ($post_id, $term_taxonomy_id, 0)";
							if ($conn->query($insert_rel_sql)) {
								// Update count
								$conn->query("UPDATE $term_taxonomy_table SET count = count + 1 WHERE term_taxonomy_id = $term_taxonomy_id");
								error_logger("Added 'affiliate' tag to post $post_id");
							}
						}
					}
				}
			}
			
			$conn->close();
			error_logger("Stored " . count($stored_ids) . " affiliate links for post $post_id");
			
			echo json_encode(['success' => true, 'stored_ids' => $stored_ids]);
			
		} catch (Exception $e) {
			error_logger("Error storing affiliate links: " . $e->getMessage());
			http_response_code(500);
			echo json_encode(['error' => 'Failed to store affiliate links: ' . $e->getMessage()]);
		}
		break;

	// GET /affiliate_links/{post_id}
	case $method === 'GET' && preg_match('#^/affiliate_links/(\d+)$#', $uri, $matches):
		require_auth();
		try {
			$post_id = (int)$matches[1];
			$active_only = isset($_GET['active_only']) ? $_GET['active_only'] === 'true' : true;
			
			error_logger("Retrieving affiliate links for post $post_id");
			$conn = get_wp_connection();
			
			if ($active_only) {
				$sql = "SELECT * FROM " . $WP_PREFIX . "ap_affiliate_links WHERE post_id = ? AND is_active = 1 ORDER BY created_at";
			} else {
				$sql = "SELECT * FROM " . $WP_PREFIX . "ap_affiliate_links WHERE post_id = ? ORDER BY created_at";
			}
			
			$stmt = $conn->prepare($sql);
			$stmt->bind_param('i', $post_id);
			$stmt->execute();
			$result = $stmt->get_result();
			
			$links = [];
			while ($row = $result->fetch_assoc()) {
				$links[] = $row;
			}
			
			$stmt->close();
			$conn->close();
			
			echo json_encode(['success' => true, 'links' => $links]);
			
		} catch (Exception $e) {
			error_logger("Error retrieving affiliate links: " . $e->getMessage());
			http_response_code(500);
			echo json_encode(['error' => 'Failed to retrieve affiliate links: ' . $e->getMessage()]);
		}
		break;

	// PUT /affiliate_links/{link_id}
	case $method === 'PUT' && preg_match('#^/affiliate_links/(\d+)$#', $uri, $matches):
		require_auth();
		try {
			$link_id = (int)$matches[1];
			$input = json_decode(file_get_contents('php://input'), true);
			
			// Check if at least one field is provided
			if (!isset($input['is_active']) && !isset($input['post_id'])) {
				http_response_code(400);
				echo json_encode(['error' => 'At least one field (is_active or post_id) is required']);
				break;
			}
			
			$conn = get_wp_connection();
			
			// Build dynamic update query
			$update_fields = [];
			$bind_types = '';
			$bind_values = [];
			
			if (isset($input['is_active'])) {
				$update_fields[] = "is_active = ?";
				$bind_types .= 'i';
				$bind_values[] = (bool)$input['is_active'];
			}
			
			if (isset($input['post_id'])) {
				$update_fields[] = "post_id = ?";
				$bind_types .= 'i';
				$bind_values[] = (int)$input['post_id'];
			}
			
			$sql = "UPDATE affiliate_links SET " . implode(', ', $update_fields) . " WHERE id = ?";
			$bind_types .= 'i';
			$bind_values[] = $link_id;
			
			error_logger("Updating affiliate link $link_id with fields: " . implode(', ', $update_fields));
			
			$stmt = $conn->prepare($sql);
			$stmt->bind_param($bind_types, ...$bind_values);
			$stmt->execute();
			
			$affected_rows = $stmt->affected_rows;
			$stmt->close();
			$conn->close();
			
			if ($affected_rows > 0) {
				echo json_encode(['success' => true, 'message' => 'Link updated successfully']);
			} else {
				http_response_code(404);
				echo json_encode(['error' => 'Link not found']);
			}
			
		} catch (Exception $e) {
			error_logger("Error updating affiliate link: " . $e->getMessage());
			http_response_code(500);
			echo json_encode(['error' => 'Failed to update affiliate link: ' . $e->getMessage()]);
		}
		break;

	// DELETE /affiliate_links/program/{program}
	case $method === 'DELETE' && preg_match('#^/affiliate_links/program/(.+)$#', $uri, $matches):
		require_auth();
		try {
			$identifier = urldecode($matches[1]);
			error_logger("Deactivating links for affiliate program: $identifier");
			
			// Resolve identifier to program name
			$resolved_program = resolve_merchant_identifier($identifier);
			if (!$resolved_program) {
				error_logger("Could not resolve merchant identifier: $identifier");
				http_response_code(400);
				echo json_encode(['error' => 'Invalid merchant identifier']);
				break;
			}
			
			$conn = get_wp_connection();
			$sql = "UPDATE affiliate_links SET is_active = 0 WHERE affiliate_program = ? AND is_active = 1";
			$stmt = $conn->prepare($sql);
			$stmt->bind_param('s', $resolved_program);
			$stmt->execute();
			
			$affected_rows = $stmt->affected_rows;
			$stmt->close();
			$conn->close();
			
			echo json_encode(['success' => true, 'message' => "Deactivated $affected_rows links for program '$resolved_program'", 'deactivated_count' => $affected_rows]);
			
		} catch (Exception $e) {
			error_logger("Error deactivating program links: " . $e->getMessage());
			http_response_code(500);
			echo json_encode(['error' => 'Failed to deactivate program links: ' . $e->getMessage()]);
		}
		break;

	// GET /affiliate_analytics
	case $method === 'GET' && preg_match('#^/affiliate_analytics/?$#', $uri):
		require_auth();
		try {
			error_logger("Generating affiliate analytics");
			$conn = get_wp_connection();
			
			$analytics = [];
			
			// Total links
			$result = $conn->query("SELECT COUNT(*) as count FROM affiliate_links");
			$analytics['total_links'] = $result->fetch_assoc()['count'];
			
			// Active links
			$result = $conn->query("SELECT COUNT(*) as count FROM affiliate_links WHERE is_active = 1");
			$analytics['active_links'] = $result->fetch_assoc()['count'];
			
			// Links by program
			$result = $conn->query("SELECT affiliate_program, COUNT(*) as count FROM affiliate_links GROUP BY affiliate_program ORDER BY count DESC");
			$analytics['links_by_program'] = [];
			while ($row = $result->fetch_assoc()) {
				$analytics['links_by_program'][] = $row;
			}
			
			// Recent links (last 30 days)
			$result = $conn->query("SELECT COUNT(*) as count FROM affiliate_links WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
			$analytics['recent_links'] = $result->fetch_assoc()['count'];
			
			$conn->close();
			echo json_encode(['success' => true, 'analytics' => $analytics]);
			
		} catch (Exception $e) {
			error_logger("Error generating analytics: " . $e->getMessage());
			http_response_code(500);
			echo json_encode(['error' => 'Failed to generate analytics: ' . $e->getMessage()]);
		}
		break;

	// DELETE /cleanup_old_links
	case $method === 'DELETE' && preg_match('#^/cleanup_old_links/?$#', $uri):
		require_auth();
		try {
			$days_old = (int)($_GET['days_old'] ?? 365);
			error_logger("Cleaning up affiliate links older than $days_old days");
			
			$conn = get_wp_connection();
			$sql = "DELETE FROM affiliate_links WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY) AND is_active = 0";
			$stmt = $conn->prepare($sql);
			$stmt->bind_param('i', $days_old);
			$stmt->execute();
			
			$affected_rows = $stmt->affected_rows;
			$stmt->close();
			$conn->close();
			
			echo json_encode(['success' => true, 'message' => "Cleaned up $affected_rows old affiliate links", 'cleaned_count' => $affected_rows]);
			
		} catch (Exception $e) {
			error_logger("Error cleaning up old links: " . $e->getMessage());
			http_response_code(500);
			echo json_encode(['error' => 'Failed to cleanup old links: ' . $e->getMessage()]);
		}
		break;

	// GET /categories-full
	case $method === 'GET' && preg_match('#^/categories-full/?$#', $uri):
		require_auth();
		try {
			// Connect to WordPress database to access Agentic Press plugin tables
			$wp = get_wp_connection();
			if (!$wp) {
				http_response_code(500);
				echo json_encode(utf8ize(['error' => 'Database connection failed']), JSON_UNESCAPED_UNICODE);
				break;
			}

			// Get all active top-level categories from Agentic Press plugin tables
			global $WP_PREFIX;
			$categories_table = $WP_PREFIX . 'ap_categories';
			$authors_table = $WP_PREFIX . 'ap_authors_categories';

			$sql = "SELECT * FROM $categories_table WHERE is_active = 1 AND (parent_id IS NULL OR parent_id = 0) ORDER BY name";
			$result = $wp->query($sql);

			if (!$result) {
				http_response_code(500);
				echo json_encode(utf8ize(['error' => 'Failed to fetch categories', 'details' => $wp->error]), JSON_UNESCAPED_UNICODE);
				$wp->close();
				break;
			}

			$categories = [];
			while ($category = $result->fetch_assoc()) {
				$category_data = build_autopress_category_structure($wp, $category, $categories_table, $authors_table);
				$categories[] = $category_data;
			}

			$result->free();
			$wp->close();
			$categories = sanitize_quotes($categories);
			$categories = utf8ize($categories);
			echo json_encode(utf8ize($categories), JSON_UNESCAPED_UNICODE);

		} catch (Exception $e) {
			error_logger("Error in /categories-full: " . $e->getMessage());
			http_response_code(500);
			echo json_encode(utf8ize(['error' => 'Internal server error']), JSON_UNESCAPED_UNICODE);
		}
		break;

	// GET /config
	case $method === 'GET' && preg_match('#^/config/?$#', $uri):
		require_auth();
		try {
			// Connect to WP DB
			$wp = get_wp_connection();
			if (!$wp) {
				http_response_code(500);
				echo json_encode(['error' => 'Database connection failed']);
				break;
			}

			// Get settings from wp_options
			$sql = "SELECT option_value FROM {$WP_PREFIX}options WHERE option_name = 'agentic_press_settings' LIMIT 1";
			$result = $wp->query($sql);
			if (!$result || $result->num_rows === 0) {
				$wp->close();
				http_response_code(404);
				echo json_encode(['error' => 'Settings not found']);
				break;
			}
			$row = $result->fetch_assoc();
			$settings = unserialize($row['option_value']);
			$wp->close();

			// Return settings (exclude sensitive data if needed)
			unset($settings['service_password']); // Don't send password back
			echo json_encode(utf8ize(['config' => $settings]), JSON_UNESCAPED_UNICODE);
		} catch (Exception $e) {
			error_logger("Error in /config: " . $e->getMessage());
			http_response_code(500);
			echo json_encode(['error' => 'Internal server error']);
		}
		break;

	default:
		http_response_code(404);
		echo json_encode(['error' => 'Endpoint not found']);
		break;
}
