<?php
// Script to fix attachment metadata for existing images
// This will update the _wp_attachment_metadata for images that don't have proper metadata

ini_set('error_log', 'fix_metadata_debug.log');
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Load config
require_once __DIR__ . '/config.php';

function get_wp_connection() {
    global $WP_HOST, $WP_USER, $WP_PASS, $WP_DB;
    $wp = new mysqli($WP_HOST, $WP_USER, $WP_PASS, $WP_DB);
    if ($wp->connect_errno) {
        die("WordPress DB connection failed: " . $wp->connect_error);
    }
    return $wp;
}

function fix_attachment_metadata() {
    $wp = get_wp_connection();

    // Find all attachment posts that are images
    $sql = "SELECT ID, guid FROM {$GLOBALS['WP_POSTS']} WHERE post_type = 'attachment' AND post_mime_type LIKE 'image/%' ORDER BY ID DESC LIMIT 50";
    $result = $wp->query($sql);

    if (!$result) {
        die("Query failed: " . $wp->error);
    }

    $fixed = 0;
    while ($row = $result->fetch_assoc()) {
        $attachment_id = $row['ID'];
        $guid = $row['guid'];

        // Check if metadata exists and is minimal
        $meta_sql = "SELECT meta_value FROM {$GLOBALS['WP_POSTMETA']} WHERE post_id = ? AND meta_key = '_wp_attachment_metadata' LIMIT 1";
        $meta_stmt = $wp->prepare($meta_sql);
        $meta_stmt->bind_param('i', $attachment_id);
        $meta_stmt->execute();
        $meta_result = $meta_stmt->get_result();

        if ($meta_result && $meta_result->num_rows > 0) {
            $meta_row = $meta_result->fetch_assoc();
            $existing_metadata = json_decode($meta_row['meta_value'], true);

            // Check if this is the old minimal metadata (just file and filesize)
            if (isset($existing_metadata['file']) && !isset($existing_metadata['width'])) {
                echo "Fixing metadata for attachment ID: $attachment_id\n";

                // Extract file path from GUID
                $parsed_url = parse_url($guid);
                $path_parts = explode('/', $parsed_url['path']);
                $filename = end($path_parts);

                // Try to find the actual file
                $upload_base = __DIR__ . '/../wp-content/uploads/';
                $year_month = date('Y/m', strtotime('now')); // This might not be accurate
                $possible_paths = [
                    $upload_base . date('Y') . '/' . date('m') . '/' . $filename,
                    $upload_base . $filename,
                    // Add more possible paths if needed
                ];

                $file_path = null;
                foreach ($possible_paths as $path) {
                    if (file_exists($path)) {
                        $file_path = $path;
                        break;
                    }
                }

                if ($file_path) {
                    $image_info = getimagesize($file_path);
                    if ($image_info) {
                        $width = $image_info[0];
                        $height = $image_info[1];
                        $file_size = filesize($file_path);

                        // Get MIME type
                        $mime_type = $image_info['mime'] ?? 'image/png';

                        // Create proper metadata
                        $metadata = [
                            'width' => $width,
                            'height' => $height,
                            'file' => $existing_metadata['file'],
                            'filesize' => $file_size,
                            'sizes' => [
                                'thumbnail' => [
                                    'file' => $filename,
                                    'width' => min($width, 150),
                                    'height' => min($height, 150),
                                    'mime-type' => $mime_type,
                                    'filesize' => $file_size
                                ],
                                'medium' => [
                                    'file' => $filename,
                                    'width' => min($width, 300),
                                    'height' => min($height, 300),
                                    'mime-type' => $mime_type,
                                    'filesize' => $file_size
                                ],
                                'large' => [
                                    'file' => $filename,
                                    'width' => min($width, 1024),
                                    'height' => min($height, 1024),
                                    'mime-type' => $mime_type,
                                    'filesize' => $file_size
                                ]
                            ],
                            'image_meta' => [
                                'aperture' => '0',
                                'credit' => '',
                                'camera' => '',
                                'caption' => '',
                                'created_timestamp' => '0',
                                'copyright' => '',
                                'focal_length' => '0',
                                'iso' => '0',
                                'shutter_speed' => '0',
                                'title' => '',
                                'orientation' => '0',
                                'keywords' => []
                            ]
                        ];

                        // Update the metadata
                        $update_sql = "UPDATE {$GLOBALS['WP_POSTMETA']} SET meta_value = ? WHERE post_id = ? AND meta_key = '_wp_attachment_metadata'";
                        $update_stmt = $wp->prepare($update_sql);
                        $update_json = json_encode($metadata);
                        $update_stmt->bind_param('si', $update_json, $attachment_id);
                        $update_stmt->execute();
                        $update_stmt->close();

                        echo "Updated metadata for attachment $attachment_id\n";
                        $fixed++;
                    } else {
                        echo "Could not get image info for $file_path\n";
                    }
                } else {
                    echo "Could not find file for attachment $attachment_id (GUID: $guid)\n";
                }
            }
        }
        $meta_stmt->close();
    }

    $result->close();
    $wp->close();

    echo "Fixed metadata for $fixed attachments\n";
}

fix_attachment_metadata();
?>