/**
 * Admin JavaScript for Author Category Ownership plugin
 */

jQuery(document).ready(function($) {
    'use strict';

    // Category form submission
    $('#ap-category-form').on('submit', function(e) {
        e.preventDefault();

        var formData = new FormData(this);
        formData.append('action', 'ap_save_category');
        formData.append('nonce', ap_ajax.nonce);

        $.ajax({
            url: ap_ajax.ajax_url,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    showMessage(response.data.message, 'success');
                    resetForm();
                    reloadCategoriesTable();
                } else {
                    showMessage(response.data.message || 'Error saving category', 'error');
                }
            },
            error: function() {
                showMessage('AJAX error occurred', 'error');
            }
        });
    });

    // Edit category
    $(document).on('click', '.ap-edit-category', function() {
        var categoryId = $(this).data('id');

        $.ajax({
            url: ap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'ap_load_category',
                category_id: categoryId,
                nonce: ap_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    populateForm(response.data);
                    $('.ap-category-form')[0].scrollIntoView({ behavior: 'smooth' });
                } else {
                    showMessage(response.data.message || 'Error loading category', 'error');
                }
            },
            error: function() {
                showMessage('AJAX error occurred', 'error');
            }
        });
    });

    // Delete category
    $(document).on('click', '.ap-delete-category', function() {
        if (!confirm('Are you sure you want to delete this category?')) {
            return;
        }

        var categoryId = $(this).data('id');

        $.ajax({
            url: ap_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'ap_delete_category',
                category_id: categoryId,
                nonce: ap_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    showMessage(response.data.message, 'success');
                    reloadCategoriesTable();
                } else {
                    showMessage(response.data.message || 'Error deleting category', 'error');
                }
            },
            error: function() {
                showMessage('AJAX error occurred', 'error');
            }
        });
    });

    // Cancel edit
    $('#ap-cancel-edit').on('click', function() {
        resetForm();
    });

    // Helper functions
    function showMessage(message, type) {
        var classes = 'notice notice-' + type + ' is-dismissible';
        var html = '<div class="' + classes + '"><p>' + message + '</p></div>';

        $('#ap-messages').html(html);

        // Auto-hide after 5 seconds
        setTimeout(function() {
            $('#ap-messages .notice').fadeOut();
        }, 5000);
    }

    function resetForm() {
        $('#ap-category-form')[0].reset();
        $('#category-id').val('');
        $('#assigned-authors-list input[type="checkbox"]').prop('checked', false);
    }

    function populateForm(data) {
        $('#category-id').val(data.id);
        $('#category-slug').val(data.category_slug);
        $('#category-name').val(data.name);
        $('#category-description').val(data.description);
        $('#system-prompt').val(data.system_prompt);
        $('#user-prompt').val(data.user_prompt);
        $('#llm-provider').val(data.llm_provider);
        $('#llm-model').val(data.model);
        $('#temperature').val(data.temperature);
        $('#top-p').val(data.top_p);
        $('#max-tokens').val(data.max_tokens);
        $('#parent-category').val(data.parent_id || '');
        $('#is-active').prop('checked', data.is_active == 1);

        // Handle assigned authors
        $('#assigned-authors-list input[type="checkbox"]').prop('checked', false);
        if (data.assigned_authors) {
            data.assigned_authors.forEach(function(authorId) {
                $('#assigned-authors-list input[value="' + authorId + '"]').prop('checked', true);
            });
        }
    }

    function reloadCategoriesTable() {
        location.reload(); // Simple reload - could be improved with AJAX table update
    }

    // Password toggle functionality for settings page
    $('#toggle_password, #toggle_api_password').on('click', function() {
        var $button = $(this);
        var $passwordField = $button.prev('input[type="password"]');

        if ($passwordField.length === 0) {
            $passwordField = $button.siblings('input[type="password"]');
        }

        if ($passwordField.attr('type') === 'password') {
            $passwordField.attr('type', 'text');
            $button.text('Hide');
        } else {
            $passwordField.attr('type', 'password');
            $button.text('Show');
        }
    });
});