<?php
/**
 * Admin interface for Author Category Ownership plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class AP_Admin {

    /**
     * Constructor
     */
    public function __construct() {
        // API Authentication menu
        add_action('admin_menu', array($this, 'add_api_auth_menu'));

        // Integration with WordPress core category pages
        add_action('category_add_form_fields', array($this, 'add_category_fields'));
        add_action('category_edit_form_fields', array($this, 'edit_category_fields'));
        add_action('created_category', array($this, 'save_category_fields'));
        add_action('edited_category', array($this, 'save_category_fields'));
        add_action('delete_term', array($this, 'delete_category_fields'), 10, 3);
        add_action('delete_user', array($this, 'delete_user_assignments'));
        add_filter('manage_edit-category_columns', array($this, 'add_category_columns'));
        add_filter('manage_category_custom_column', array($this, 'manage_category_columns'), 10, 3);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_category_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_plugin_styles'));

        // Initialize settings
        new AP_Settings();
    }

    /**
     * Add API Authentication menu
     */
    public function add_api_auth_menu() {
        add_menu_page(
            __('Agentic Press', 'agentic-press'),
            __('Agentic Press', 'agentic-press'),
            'manage_options',
            'agentic-press-main',
            array($this, 'agentic_press_main_page'),
            'dashicons-admin-tools',
            30
        );
        add_submenu_page(
            'agentic-press-main',
            __('Dashboard', 'agentic-press'),
            __('Dashboard', 'agentic-press'),
            'manage_options',
            'agentic-press-main',
            array($this, 'agentic_press_main_page')
        );
        add_submenu_page(
            'agentic-press-main',
            __('Settings', 'agentic-press'),
            __('Settings', 'agentic-press'),
            'manage_options',
            'agentic-press-settings',
            array($this, 'render_settings_page')
        );
        add_submenu_page(
            'agentic-press-main',
            __('Categories', 'agentic-press'),
            __('Categories', 'agentic-press'),
            'manage_options',
            'agentic-press-categories',
            array($this, 'categories_page')
        );
        add_submenu_page(
            'agentic-press-main',
            __('API Authentication', 'agentic-press'),
            __('API Authentication', 'agentic-press'),
            'manage_options',
            'agentic-press-api-auth',
            array($this, 'api_auth_page')
        );
        add_submenu_page(
            'agentic-press-main',
            __('Affiliate Management', 'agentic-press'),
            __('Affiliate Management', 'agentic-press'),
            'manage_options',
            'agentic-press-affiliate',
            array($this, 'affiliate_management_page')
        );
    }

    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts($hook) {
        // Only load scripts for category pages now
        if ($hook === 'edit-tags.php' && isset($_GET['taxonomy']) && $_GET['taxonomy'] === 'category') {
            wp_enqueue_style('ap-admin-css', AP_PLUGIN_URL . 'assets/css/admin.css', array(), AP_VERSION);
        }
    }

    /**
     * API Authentication page
     */
    public function api_auth_page() {
        $user = wp_get_current_user();
        if (!in_array('administrator', $user->roles)) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }

        // Handle form submissions
        if (isset($_POST['add_api_user']) && check_admin_referer('add_api_user', 'api_user_nonce')) {
            $this->handle_add_api_user();
        }

        if (isset($_POST['delete_api_user']) && check_admin_referer('delete_api_user', 'delete_user_nonce')) {
            $this->handle_delete_api_user();
        }

        ?>
        <div class="wrap">
            <h1><?php _e('API Authentication', 'agentic-press'); ?></h1>

            <div id="api-messages"></div>

            <div class="api-user-form">
                <h2><?php _e('Add API User', 'agentic-press'); ?></h2>
                <form method="post">
                    <?php wp_nonce_field('add_api_user', 'api_user_nonce'); ?>
                    <table class="form-table">
                        <tr>
                            <th><label for="api_username"><?php _e('Username', 'agentic-press'); ?></label></th>
                            <td><input type="text" id="api_username" name="api_username" class="regular-text" required></td>
                        </tr>
                        <tr>
                            <th><label for="api_password"><?php _e('Password', 'agentic-press'); ?></label></th>
                            <td>
                                <input type="password" id="api_password" name="api_password" class="regular-text" required>
                                <button type="button" id="toggle_api_password" class="button button-small"><?php _e('Show', 'agentic-press'); ?></button>
                            </td>
                        </tr>
                    </table>
                    <p class="submit">
                        <input type="submit" name="add_api_user" class="button button-primary" value="<?php _e('Add API User', 'agentic-press'); ?>">
                    </p>
                </form>
                <script>
                document.getElementById('toggle_api_password').addEventListener('click', function() {
                    var passwordField = document.getElementById('api_password');
                    var button = this;
                    
                    if (passwordField.type === 'password') {
                        passwordField.type = 'text';
                        button.textContent = '<?php _e('Hide', 'agentic-press'); ?>';
                    } else {
                        passwordField.type = 'password';
                        button.textContent = '<?php _e('Show', 'agentic-press'); ?>';
                    }
                });
                </script>
            </div>

            <div class="api-users-list">
                <h2><?php _e('API Users', 'autopress'); ?></h2>
                <?php $this->render_api_users_table(); ?>
            </div>
        </div>
        <?php
    }

    /**
     * Handle adding API user
     */
    private function handle_add_api_user() {
        global $wpdb;
        $table_users = $wpdb->prefix . 'ap_users';

        $username = sanitize_text_field($_POST['api_username']);
        $password = $_POST['api_password'];

        if (empty($username) || empty($password)) {
            echo '<div class="notice notice-error"><p>' . __('Username and password are required.', 'agentic-press') . '</p></div>';
            return;
        }

        // Check if username exists
        $existing = $wpdb->get_var($wpdb->prepare("SELECT id FROM $table_users WHERE username = %s", $username));
        if ($existing) {
            echo '<div class="notice notice-error"><p>' . __('Username already exists.', 'agentic-press') . '</p></div>';
            return;
        }

        // Hash password
        $password_hash = password_hash($password, PASSWORD_DEFAULT);

        // Insert user
        $result = $wpdb->insert(
            $table_users,
            array(
                'username' => $username,
                'password_hash' => $password_hash
            ),
            array('%s', '%s')
        );

        if ($result) {
            echo '<div class="notice notice-success"><p>' . __('API user added successfully.', 'agentic-press') . '</p></div>';
        } else {
            echo '<div class="notice notice-error"><p>' . __('Failed to add API user.', 'agentic-press') . '</p></div>';
        }
    }

    /**
     * Handle deleting API user
     */
    private function handle_delete_api_user() {
        global $wpdb;
        $table_users = $wpdb->prefix . 'ap_users';
        $table_tokens = $wpdb->prefix . 'ap_tokens';

        $user_id = intval($_POST['user_id']);

        // Delete tokens first
        $wpdb->delete($table_tokens, array('user_id' => $user_id), array('%d'));

        // Delete user
        $result = $wpdb->delete($table_users, array('id' => $user_id), array('%d'));

        if ($result) {
            echo '<div class="notice notice-success"><p>' . __('API user deleted successfully.', 'agentic-press') . '</p></div>';
        } else {
            echo '<div class="notice notice-error"><p>' . __('Failed to delete API user.', 'agentic-press') . '</p></div>';
        }
    }

    /**
     * Render API users table
     */
    private function render_api_users_table() {
        global $wpdb;
        $table_users = $wpdb->prefix . 'ap_users';

        $users = $wpdb->get_results("SELECT id, username, created_at FROM $table_users ORDER BY created_at DESC");

        if (empty($users)) {
            echo '<p>' . __('No API users found.', 'agentic-press') . '</p>';
            return;
        }

        echo '<table class="wp-list-table widefat fixed striped">';
        echo '<thead><tr><th>' . __('Username', 'agentic-press') . '</th><th>' . __('Created', 'agentic-press') . '</th><th>' . __('Actions', 'agentic-press') . '</th></tr></thead>';
        echo '<tbody>';

        foreach ($users as $user) {
            echo '<tr>';
            echo '<td>' . esc_html($user->username) . '</td>';
            echo '<td>' . esc_html($user->created_at) . '</td>';
            echo '<td>';
            echo '<form method="post" style="display:inline;">';
            wp_nonce_field('delete_api_user', 'delete_user_nonce');
            echo '<input type="hidden" name="user_id" value="' . esc_attr($user->id) . '">';
            echo '<input type="submit" name="delete_api_user" class="button button-small button-link-delete" value="' . __('Delete', 'agentic-press') . '" onclick="return confirm(\'' . __('Are you sure you want to delete this API user?', 'agentic-press') . '\');">';
            echo '</form>';
            echo '</td>';
            echo '</tr>';
        }

        echo '</tbody></table>';
    }

    /**
     * Update author assignments for a category
     */
    private function update_author_assignments($category_id, $assigned_authors) {
        global $wpdb;
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';

        // Remove all existing assignments for this category
        $wpdb->delete($table_authors_categories, array('category_id' => $category_id), array('%d'));

        // Add new assignments
        foreach ($assigned_authors as $author_id) {
            $wpdb->insert(
                $table_authors_categories,
                array('category_id' => $category_id, 'author_id' => $author_id),
                array('%d', '%d')
            );
        }
    }

    /**
     * Add custom fields to add category form
     */
    public function add_category_fields() {
        ?>
        <div class="form-field">
            <label for="ap_system_prompt"><?php _e('System Prompt', 'autopress'); ?></label>
            <textarea id="ap_system_prompt" name="ap_system_prompt" rows="5" class="large-text"></textarea>
            <p><?php _e('AI system prompt for content generation in this category.', 'autopress'); ?></p>
        </div>

        <div class="form-field">
            <label for="ap_user_prompt"><?php _e('User Prompt', 'autopress'); ?></label>
            <textarea id="ap_user_prompt" name="ap_user_prompt" rows="5" class="large-text"></textarea>
            <p><?php _e('AI user prompt template for content generation in this category.', 'autopress'); ?></p>
        </div>

        <div class="form-field">
            <label for="ap_llm_provider"><?php _e('LLM Provider', 'autopress'); ?></label>
            <input type="text" id="ap_llm_provider" name="ap_llm_provider" value="openai" class="regular-text">
            <p><?php _e('AI provider (e.g., openai, anthropic).', 'autopress'); ?></p>
        </div>

        <div class="form-field">
            <label for="ap_llm_model"><?php _e('Model', 'autopress'); ?></label>
            <input type="text" id="ap_llm_model" name="ap_llm_model" value="o4-mini" class="regular-text">
            <p><?php _e('AI model to use for content generation.', 'autopress'); ?></p>
        </div>

        <div class="form-field">
            <label for="ap_temperature"><?php _e('Temperature', 'autopress'); ?></label>
            <input type="number" id="ap_temperature" name="ap_temperature" step="0.1" min="0" max="2" value="0.5" class="small-text">
            <p><?php _e('AI creativity level (0.0-2.0).', 'autopress'); ?></p>
        </div>

        <div class="form-field">
            <label for="ap_top_p"><?php _e('Top P', 'autopress'); ?></label>
            <input type="number" id="ap_top_p" name="ap_top_p" step="0.1" min="0" max="1" value="0.8" class="small-text">
            <p><?php _e('AI nucleus sampling parameter (0.0-1.0).', 'autopress'); ?></p>
        </div>

        <div class="form-field">
            <label for="ap_max_tokens"><?php _e('Max Tokens', 'autopress'); ?></label>
            <input type="number" id="ap_max_tokens" name="ap_max_tokens" value="10000" class="small-text">
            <p><?php _e('Maximum tokens for AI response.', 'autopress'); ?></p>
        </div>

        <div class="form-field">
            <label for="ap_parent_category"><?php _e('Parent Category', 'autopress'); ?></label>
            <select id="ap_parent_category" name="ap_parent_category">
                <option value=""><?php _e('None (Top Level)', 'autopress'); ?></option>
                <?php
                $categories = AP_DB::get_all_categories();
                foreach ($categories as $category) {
                    echo '<option value="' . esc_attr($category->id) . '">' . esc_html($category->name) . '</option>';
                }
                ?>
            </select>
            <p><?php _e('Parent Agentic Press category for hierarchical organization.', 'autopress'); ?></p>
        </div>

        <div class="form-field">
            <label for="ap_is_active"><?php _e('Active', 'autopress'); ?></label>
            <input type="checkbox" id="ap_is_active" name="ap_is_active" checked>
            <p><?php _e('Enable this category for Agentic Press content generation.', 'autopress'); ?></p>
        </div>

        <div class="form-field">
            <label for="ap_assigned_authors"><?php _e('Assigned Authors', 'autopress'); ?></label>
            <div id="ap-category-authors">
                <?php
                $users = get_users(array('role__in' => array('author', 'editor', 'administrator')));
                foreach ($users as $user) {
                    echo '<label style="display: block; margin-bottom: 5px;"><input type="checkbox" name="ap_assigned_authors[]" value="' . esc_attr($user->ID) . '"> ' . esc_html($user->display_name) . ' (' . esc_html($user->user_email) . ')</label>';
                }
                ?>
            </div>
            <p><?php _e('Select authors to assign to this category. Authors can only be assigned to one category.', 'autopress'); ?></p>
        </div>
        <?php
    }

    /**
     * Add custom fields to edit category form
     */
    public function edit_category_fields($term) {
        $category_slug = $term->slug;
        $ap_category = AP_DB::get_category_by_slug($category_slug);
        $assigned_authors = array();

        if ($ap_category) {
            $assigned_authors = AP_DB::get_authors_by_category($category_slug);
            $assigned_authors = wp_list_pluck($assigned_authors, 'ID');
        }
        ?>
        <tr class="form-field">
            <th scope="row"><label for="ap_system_prompt"><?php _e('System Prompt', 'autopress'); ?></label></th>
            <td>
                <textarea id="ap_system_prompt" name="ap_system_prompt" rows="5" class="large-text"><?php echo esc_textarea($ap_category ? ($ap_category->system_prompt ?? '') : ''); ?></textarea>
                <p class="description"><?php _e('AI system prompt for content generation in this category.', 'autopress'); ?></p>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row"><label for="ap_user_prompt"><?php _e('User Prompt', 'autopress'); ?></label></th>
            <td>
                <textarea id="ap_user_prompt" name="ap_user_prompt" rows="5" class="large-text"><?php echo esc_textarea($ap_category ? $ap_category->user_prompt : ''); ?></textarea>
                <p class="description"><?php _e('AI user prompt template for content generation in this category.', 'autopress'); ?></p>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row"><label for="ap_llm_provider"><?php _e('LLM Provider', 'autopress'); ?></label></th>
            <td>
                <input type="text" id="ap_llm_provider" name="ap_llm_provider" value="<?php echo esc_attr($ap_category ? $ap_category->llm_provider : 'openai'); ?>" class="regular-text">
                <p class="description"><?php _e('AI provider (e.g., openai, anthropic).', 'autopress'); ?></p>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row"><label for="ap_llm_model"><?php _e('Model', 'autopress'); ?></label></th>
            <td>
                <input type="text" id="ap_llm_model" name="ap_llm_model" value="<?php echo esc_attr($ap_category ? $ap_category->model : 'o4-mini'); ?>" class="regular-text">
                <p class="description"><?php _e('AI model to use for content generation.', 'autopress'); ?></p>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row"><label for="ap_temperature"><?php _e('Temperature', 'autopress'); ?></label></th>
            <td>
                <input type="number" id="ap_temperature" name="ap_temperature" step="0.1" min="0" max="2" value="<?php echo esc_attr($ap_category ? $ap_category->temperature : '0.5'); ?>" class="small-text">
                <p class="description"><?php _e('AI creativity level (0.0-2.0).', 'autopress'); ?></p>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row"><label for="ap_top_p"><?php _e('Top P', 'autopress'); ?></label></th>
            <td>
                <input type="number" id="ap_top_p" name="ap_top_p" step="0.1" min="0" max="1" value="<?php echo esc_attr($ap_category ? $ap_category->top_p : '0.8'); ?>" class="small-text">
                <p class="description"><?php _e('AI nucleus sampling parameter (0.0-1.0).', 'autopress'); ?></p>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row"><label for="ap_max_tokens"><?php _e('Max Tokens', 'autopress'); ?></label></th>
            <td>
                <input type="number" id="ap_max_tokens" name="ap_max_tokens" value="<?php echo esc_attr($ap_category ? $ap_category->max_tokens : '10000'); ?>" class="small-text">
                <p class="description"><?php _e('Maximum tokens for AI response.', 'autopress'); ?></p>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row"><label for="ap_parent_category"><?php _e('Parent Category', 'autopress'); ?></label></th>
            <td>
                <select id="ap_parent_category" name="ap_parent_category">
                    <option value=""><?php _e('None (Top Level)', 'autopress'); ?></option>
                    <?php
                    $categories = AP_DB::get_all_categories();
                    foreach ($categories as $category) {
                        if ($ap_category && $category->id == $ap_category->id) continue; // Don't allow self-reference
                        $selected = ($ap_category && $ap_category->parent_id == $category->id) ? 'selected' : '';
                        echo '<option value="' . esc_attr($category->id) . '" ' . $selected . '>' . esc_html($category->name) . '</option>';
                    }
                    ?>
                </select>
                <p class="description"><?php _e('Parent Agentic Press category for hierarchical organization.', 'autopress'); ?></p>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row"><label for="ap_is_active"><?php _e('Active', 'autopress'); ?></label></th>
            <td>
                <input type="checkbox" id="ap_is_active" name="ap_is_active" <?php checked($ap_category ? $ap_category->is_active : true); ?>>
                <p class="description"><?php _e('Enable this category for Agentic Press content generation.', 'autopress'); ?></p>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row"><label for="ap_assigned_authors"><?php _e('Assigned Authors', 'autopress'); ?></label></th>
            <td>
                <div id="ap-category-authors">
                    <?php
                    $users = get_users(array('role__in' => array('author', 'editor', 'administrator')));
                    foreach ($users as $user) {
                        $checked = in_array($user->ID, $assigned_authors) ? 'checked' : '';
                        echo '<label style="display: block; margin-bottom: 5px;"><input type="checkbox" name="ap_assigned_authors[]" value="' . esc_attr($user->ID) . '" ' . $checked . '> ' . esc_html($user->display_name) . ' (' . esc_html($user->user_email) . ')</label>';
                    }
                    ?>
                </div>
                <p class="description"><?php _e('Select authors to assign to this category.', 'autopress'); ?></p>
            </td>
        </tr>
        <?php
    }

    /**
     * Save custom category fields
     */
    public function save_category_fields($term_id) {
        if (!current_user_can('manage_categories')) {
            return;
        }

        $term = get_term($term_id, 'category');
        if (!$term || is_wp_error($term)) {
            return;
        }

        $assigned_authors = isset($_POST['ap_assigned_authors']) ? array_map('intval', $_POST['ap_assigned_authors']) : array();

        // Get or create AP category
        $ap_category = AP_DB::get_category_by_slug($term->slug);

        // If this is a new AP category and no authors were selected, try to get authors from JSON
        if (!$ap_category && empty($assigned_authors)) {
            $assigned_authors = $this->get_authors_from_json($term->slug);
        }

        global $wpdb;
        $table = $wpdb->prefix . AP_TABLE_CATEGORIES;

        $category_data = array(
            'term_id' => $term_id,
            'category_slug' => $term->slug,
            'name' => $term->name,
            'description' => $term->description,
            'system_prompt' => wp_unslash($_POST['ap_system_prompt'] ?? ''),
            'user_prompt' => wp_unslash($_POST['ap_user_prompt'] ?? ''),
            'llm_provider' => sanitize_text_field($_POST['ap_llm_provider'] ?? 'openai'),
            'model' => sanitize_text_field($_POST['ap_llm_model'] ?? 'o4-mini'),
            'temperature' => floatval($_POST['ap_temperature'] ?? 0.5),
            'top_p' => floatval($_POST['ap_top_p'] ?? 0.8),
            'max_tokens' => intval($_POST['ap_max_tokens'] ?? 10000),
            'parent_id' => !empty($_POST['ap_parent_category']) ? intval($_POST['ap_parent_category']) : null,
            'is_active' => isset($_POST['ap_is_active']) ? 1 : 0
        );

        if (!$ap_category) {
            // Create new AP category entry
            $wpdb->insert($table, $category_data);
            $ap_category_id = $wpdb->insert_id;
        } else {
            // Update existing AP category entry
            $wpdb->update($table, $category_data, array('id' => $ap_category->id));
            $ap_category_id = $ap_category->id;
        }

        // Update author assignments
        $this->update_author_assignments($ap_category_id, $assigned_authors);
    }

    /**
     * Delete category fields when a category is deleted
     */
    public function delete_category_fields($term_id, $tt_id, $taxonomy) {
        if ($taxonomy !== 'category') {
            return;
        }

        global $wpdb;
        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';

        // Get the AP category by term_id
        $ap_category = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $table_categories WHERE term_id = %d",
            $term_id
        ));

        if ($ap_category) {
            // Delete from authors-categories table
            $wpdb->delete($table_authors_categories, array('category_id' => $ap_category->id), array('%d'));

            // Update child categories to remove parent reference
            $wpdb->update($table_categories, array('parent_id' => null), array('parent_id' => $ap_category->id), array('%s'), array('%d'));

            // Delete from categories table
            $wpdb->delete($table_categories, array('id' => $ap_category->id), array('%d'));
        }
    }

    /**
     * Delete user assignments when a user is deleted
     */
    public function delete_user_assignments($user_id) {
        global $wpdb;
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';

        // Delete all category assignments for this user
        $wpdb->delete($table_authors_categories, array('author_id' => $user_id), array('%d'));
    }

    /**
     * Get authors from JSON file for a category slug
     */
    private function get_authors_from_json($category_slug) {
        $json_file = AP_PLUGIN_DIR . 'categories.json';

        if (!file_exists($json_file)) {
            return array();
        }

        $json_content = file_get_contents($json_file);
        if ($json_content === false) {
            return array();
        }

        $categories = json_decode($json_content, true);
        if (!$categories) {
            return array();
        }

        // Find the category in JSON
        foreach ($categories as $category) {
            if ($category['category'] === $category_slug) {
                $authors = array();

                // Extract authors from envs structure
                if (isset($category['authors']['envs'])) {
                    foreach ($category['authors']['envs'] as $env) {
                        if (isset($env['authors'])) {
                            foreach ($env['authors'] as $author) {
                                if (isset($author['id'])) {
                                    $authors[] = intval($author['id']);
                                }
                            }
                        }
                    }
                }

                return $authors;
            }

            // Check subcategories recursively
            if (isset($category['sub_categories']) && is_array($category['sub_categories'])) {
                $sub_authors = $this->get_authors_from_subcategories($category['sub_categories'], $category_slug);
                if (!empty($sub_authors)) {
                    return $sub_authors;
                }
            }
        }

        return array();
    }

    /**
     * Recursively search subcategories for authors
     */
    private function get_authors_from_subcategories($subcategories, $category_slug) {
        foreach ($subcategories as $subcategory) {
            if ($subcategory['category'] === $category_slug) {
                $authors = array();

                if (isset($subcategory['authors']['envs'])) {
                    foreach ($subcategory['authors']['envs'] as $env) {
                        if (isset($env['authors'])) {
                            foreach ($env['authors'] as $author) {
                                if (isset($author['id'])) {
                                    $authors[] = intval($author['id']);
                                }
                            }
                        }
                    }
                }

                return $authors;
            }

            // Check deeper subcategories
            if (isset($subcategory['sub_categories']) && is_array($subcategory['sub_categories'])) {
                $deep_authors = $this->get_authors_from_subcategories($subcategory['sub_categories'], $category_slug);
                if (!empty($deep_authors)) {
                    return $deep_authors;
                }
            }
        }

        return array();
    }

    /**
     * Add custom columns to category list table
     */
    public function add_category_columns($columns) {
        $columns['ap_assigned_authors'] = __('Assigned Authors', 'autopress');
        return $columns;
    }

    /**
     * Manage custom columns in category list table
     */
    public function manage_category_columns($content, $column_name, $term_id) {
        if ($column_name === 'ap_assigned_authors') {
            $term = get_term($term_id, 'category');
            if ($term && !is_wp_error($term)) {
                $ap_category = AP_DB::get_category_by_slug($term->slug);
                if ($ap_category) {
                    $assigned_authors = AP_DB::get_authors_by_category($term->slug);
                    if (!empty($assigned_authors)) {
                        $author_names = array();
                        foreach ($assigned_authors as $author) {
                            $author_names[] = $author->display_name;
                        }
                        $content = implode(', ', $author_names);
                    } else {
                        $content = __('None', 'autopress');
                    }
                } else {
                    $content = __('None', 'autopress');
                }
            }
        }
        return $content;
    }

    /**
     * Enqueue scripts for category pages
     */
    public function enqueue_category_scripts($hook) {
        if ($hook === 'edit-tags.php' && isset($_GET['taxonomy']) && $_GET['taxonomy'] === 'category') {
            wp_enqueue_style('ap-admin-css', AP_PLUGIN_URL . 'assets/css/admin.css', array(), AP_VERSION);
        }
    }

    /**
     * Enqueue plugin-specific styles for admin pages
     */
    public function enqueue_plugin_styles($hook) {
        // Enqueue the plugin's custom style.css on all admin pages for testing
        // wp_enqueue_style('agentic-press-custom-css', AP_PLUGIN_URL . 'assets/css/style.css', array(), AP_VERSION);
    }

    /**
     * Render the settings page using AP_Settings
     */
    public function render_settings_page() {
        $settings = new AP_Settings();
        $settings->render_settings_page();
    }

    /**
     * Render the main dashboard page
     */
    public function agentic_press_main_page() {
        $user = wp_get_current_user();
        if (!in_array('administrator', $user->roles)) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }

        $settings = new AP_Settings();
        $plugin_settings = $settings->get_settings();

        ?>
        <div class="wrap">
            <h1><?php _e('Agentic Press Dashboard', 'agentic-press'); ?></h1>

            <div class="agentic-press-dashboard">
                <div class="dashboard-section">
                    <h2><?php _e('Plugin Status', 'agentic-press'); ?></h2>
                    <div class="status-grid">
                        <div class="status-item">
                            <h3><?php _e('Service Connection', 'agentic-press'); ?></h3>
                            <span class="status-indicator <?php echo !empty($plugin_settings['service_username']) && !empty($plugin_settings['service_password']) ? 'status-active' : 'status-inactive'; ?>">
                                <?php echo !empty($plugin_settings['service_username']) && !empty($plugin_settings['service_password']) ? __('Connected', 'agentic-press') : __('Not Connected', 'agentic-press'); ?>
                            </span>
                        </div>
                        <div class="status-item">
                            <h3><?php _e('API Authentication', 'agentic-press'); ?></h3>
                            <span class="status-indicator <?php echo !empty($plugin_settings['api_auth_username']) && !empty($plugin_settings['api_auth_password']) ? 'status-active' : 'status-inactive'; ?>">
                                <?php echo !empty($plugin_settings['api_auth_username']) && !empty($plugin_settings['api_auth_password']) ? __('Configured', 'agentic-press') : __('Not Configured', 'agentic-press'); ?>
                            </span>
                        </div>
                        <div class="status-item">
                            <h3><?php _e('AI Provider', 'agentic-press'); ?></h3>
                            <span class="status-indicator status-active">
                                <?php echo ucfirst($plugin_settings['llm_provider'] ?? 'openai'); ?>
                            </span>
                        </div>
                    </div>
                </div>

                <div class="dashboard-section">
                    <h2><?php _e('Quick Actions', 'agentic-press'); ?></h2>
                    <div class="action-buttons">
                        <a href="<?php echo admin_url('admin.php?page=agentic-press-settings'); ?>" class="button button-primary">
                            <?php _e('Configure Settings', 'agentic-press'); ?>
                        </a>
                        <a href="<?php echo admin_url('admin.php?page=agentic-press-api-auth'); ?>" class="button button-secondary">
                            <?php _e('Manage API Users', 'agentic-press'); ?>
                        </a>
                        <a href="<?php echo admin_url('edit-tags.php?taxonomy=category'); ?>" class="button button-secondary">
                            <?php _e('Manage Categories', 'agentic-press'); ?>
                        </a>
                    </div>
                </div>

                <div class="dashboard-section">
                    <h2><?php _e('Getting Started', 'agentic-press'); ?></h2>
                    <div class="getting-started">
                        <ol>
                            <li><?php _e('Configure your Agentic Press service credentials in Settings', 'agentic-press'); ?></li>
                            <li><?php _e('Create API users for WordPress authentication', 'agentic-press'); ?></li>
                            <li><?php _e('Set up your AI provider API keys', 'agentic-press'); ?></li>
                            <li><?php _e('Configure content generation preferences', 'agentic-press'); ?></li>
                            <li><?php _e('Assign authors to categories for automated publishing', 'agentic-press'); ?></li>
                        </ol>
                    </div>
                </div>
            </div>

            <style>
                .agentic-press-dashboard {
                    margin-top: 20px;
                }
                .dashboard-section {
                    background: #fff;
                    border: 1px solid #ccd0d4;
                    border-radius: 4px;
                    margin-bottom: 20px;
                    padding: 20px;
                }
                .dashboard-section h2 {
                    margin-top: 0;
                    color: #23282d;
                }
                .status-grid {
                    display: grid;
                    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                    gap: 15px;
                    margin-top: 15px;
                }
                .status-item {
                    padding: 15px;
                    background: #f8f9fa;
                    border-radius: 4px;
                    text-align: center;
                }
                .status-item h3 {
                    margin: 0 0 10px 0;
                    font-size: 14px;
                    color: #666;
                }
                .status-indicator {
                    display: inline-block;
                    padding: 4px 8px;
                    border-radius: 12px;
                    font-size: 12px;
                    font-weight: 500;
                }
                .status-active {
                    background: #d4edda;
                    color: #155724;
                }
                .status-inactive {
                    background: #f8d7da;
                    color: #721c24;
                }
                .action-buttons {
                    margin-top: 15px;
                }
                .action-buttons .button {
                    margin-right: 10px;
                    margin-bottom: 10px;
                }
                .getting-started ol {
                    margin: 15px 0 0 20px;
                }
                .getting-started li {
                    margin-bottom: 8px;
                    line-height: 1.4;
                }
            </style>
        </div>
        <?php
    }

    /**
     * Categories page for export/import functionality
     */
    public function categories_page() {
        // Handle export
        if (isset($_POST['export_categories'])) {
            $this->export_categories();
            return;
        }

        // Handle import
        if (isset($_POST['import_categories']) && isset($_FILES['categories_file'])) {
            $this->import_categories();
            return;
        }

        ?>
        <div class="wrap">
            <h1><?php _e('Agentic Press - Categories', 'agentic-press'); ?></h1>

            <?php if (isset($_GET['imported'])): ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php _e('Categories imported successfully!', 'agentic-press'); ?></p>
                </div>
            <?php endif; ?>

            <div class="card">
                <h2><?php _e('Export Categories', 'agentic-press'); ?></h2>
                <p><?php _e('Export your current categories configuration to a JSON file. This file can be imported into another Agentic Press installation.', 'agentic-press'); ?></p>
                <form method="post">
                    <?php wp_nonce_field('ap_export_categories', 'ap_export_nonce'); ?>
                    <p>
                        <input type="submit" name="export_categories" class="button button-primary" value="<?php _e('Export Categories', 'agentic-press'); ?>">
                    </p>
                </form>
            </div>

            <div class="card">
                <h2><?php _e('Import Categories', 'agentic-press'); ?></h2>
                <p><?php _e('Import categories from a JSON file. This will replace your current categories configuration.', 'agentic-press'); ?></p>
                <form method="post" enctype="multipart/form-data">
                    <?php wp_nonce_field('ap_import_categories', 'ap_import_nonce'); ?>
                    <p>
                        <input type="file" name="categories_file" accept=".json" required>
                    </p>
                    <p>
                        <label>
                            <input type="checkbox" name="confirm_import" required>
                            <?php _e('I understand this will replace all existing categories', 'agentic-press'); ?>
                        </label>
                    </p>
                    <p>
                        <input type="submit" name="import_categories" class="button button-secondary" value="<?php _e('Import Categories', 'agentic-press'); ?>">
                    </p>
                </form>
            </div>
        </div>
        <?php
    }

    /**
     * Export categories to JSON file
     */
    private function export_categories() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['ap_export_nonce'], 'ap_export_categories')) {
            wp_die(__('Security check failed', 'agentic-press'));
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'agentic-press'));
        }

        global $wpdb;
        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';

        // Get all categories with their hierarchical structure
        $categories = $this->get_categories_hierarchy();

        // Set headers for file download
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="agentic-press-categories-' . date('Y-m-d-H-i-s') . '.json"');
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');

        // Output JSON
        echo json_encode($categories, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }

    /**
     * Import categories from JSON file
     */
    private function import_categories() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['ap_import_nonce'], 'ap_import_categories')) {
            wp_die(__('Security check failed', 'agentic-press'));
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'agentic-press'));
        }

        // Check if file was uploaded
        if (!isset($_FILES['categories_file']) || $_FILES['categories_file']['error'] !== UPLOAD_ERR_OK) {
            wp_die(__('File upload failed', 'agentic-press'));
        }

        // Read and decode JSON
        $json_content = file_get_contents($_FILES['categories_file']['tmp_name']);
        $categories = json_decode($json_content, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            wp_die(__('Invalid JSON file', 'agentic-press'));
        }

        // Import categories
        AP_DB::migrate_categories_json_from_array($categories);

        // Redirect back with success message
        wp_redirect(add_query_arg('imported', '1', wp_get_referer()));
        exit;
    }

    /**
     * Get categories hierarchy for export
     */
    private function get_categories_hierarchy($parent_id = null) {
        global $wpdb;
        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';

        $query = $wpdb->prepare("
            SELECT c.* FROM $table_categories c
            WHERE c.parent_id " . ($parent_id === null ? "IS NULL" : "= %d") . "
            AND c.is_active = 1
            ORDER BY c.name ASC
        ", $parent_id);

        $categories = $wpdb->get_results($query, ARRAY_A);

        $result = array();
        foreach ($categories as $category) {
            // Get authors for this category
            $authors_query = $wpdb->prepare("
                SELECT u.ID, u.display_name
                FROM $table_authors_categories ac
                JOIN {$wpdb->users} u ON ac.author_id = u.ID
                WHERE ac.category_id = %d
            ", $category['id']);

            $authors = $wpdb->get_results($authors_query, ARRAY_A);

            // Build category structure
            $category_data = array(
                'category' => $category['category_slug'],
                'name' => $category['name'],
                'description' => $category['description'],
                'system_prompt' => $category['system_prompt'],
                'user_prompt' => $category['user_prompt'],
                'llm_provider' => $category['llm_provider'],
                'model' => $category['model'],
                'temperature' => floatval($category['temperature']),
                'top_p' => floatval($category['top_p']),
                'max_tokens' => intval($category['max_tokens']),
                'include_base_system_prompt' => (bool)$category['include_base_system_prompt'],
                'authors' => array(
                    'envs' => array(
                        array(
                            'env' => 'prod',
                            'authors' => array_map(function($author) {
                                return array(
                                    'role' => 'author',
                                    'name' => $author['display_name'],
                                    'id' => intval($author['ID'])
                                );
                            }, $authors)
                        ),
                        array(
                            'env' => 'lab',
                            'authors' => array_map(function($author) {
                                return array(
                                    'role' => 'author',
                                    'name' => $author['display_name'],
                                    'id' => intval($author['ID'])
                                );
                            }, $authors)
                        )
                    )
                ),
                'sub_categories' => $this->get_categories_hierarchy($category['id'])
            );

            $result[] = $category_data;
        }

        return $result;
    }

    /**
     * Affiliate Management Page
     */
    public function affiliate_management_page() {
        // Handle form submissions
        if (isset($_POST['update_merchants'])) {
            $this->handle_update_merchants($_POST['merchants'] ?? array());
        }

        if (isset($_POST['delete_merchants']) && isset($_POST['merchants_to_delete'])) {
            $this->handle_delete_merchants($_POST['merchants_to_delete']);
        }

        // Get all merchants
        $merchants = $this->get_affiliate_merchants();

        ?>
        <div class="wrap">
            <h1><?php _e('Affiliate Management', 'agentic-press'); ?></h1>

            <?php settings_errors('affiliate_management'); ?>

            <div class="notice notice-info">
                <p><?php _e('Manage affiliate programs and their link visibility. Check the boxes for programs you want to enable (unchecked programs will be disabled). Use the delete section below for permanent removal.', 'agentic-press'); ?></p>
            </div>

            <!-- Update Merchants Section -->
            <div class="affiliate-section">
                <h2><?php _e('Update Program Status', 'agentic-press'); ?></h2>
                <p><?php _e('Check the affiliate programs you want to enable (unchecked programs will be disabled). Click Update Status to apply changes.', 'agentic-press'); ?></p>

                <form method="post" action="">
                    <?php wp_nonce_field('update_merchants_action', 'update_merchants_nonce'); ?>

                    <div class="merchants-list-section">
                        <h3><?php _e('Current Program Status', 'agentic-press'); ?></h3>
                        <div class="merchant-checkboxes">
                            <?php foreach ($merchants as $merchant): ?>
                                <label class="merchant-checkbox-item">
                                    <input type="checkbox"
                                           name="merchants[]"
                                           value="<?php echo esc_attr($merchant['affiliate_program']); ?>"
                                           <?php checked($merchant['is_active'], 1); ?> />
                                    <span class="merchant-info">
                                        <strong><?php echo esc_html($merchant['affiliate_program']); ?></strong>
                                        <span class="link-count">(<?php echo intval($merchant['link_count']); ?> links)</span>
                                    </span>
                                </label>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <div class="action-controls">
                        <button type="submit" name="update_merchants" class="button button-primary">
                            <?php _e('Update Status', 'agentic-press'); ?>
                        </button>
                    </div>
                </form>
            </div>

            <!-- Delete Merchants Section -->
            <div class="affiliate-section affiliate-delete-section">
                <h2><?php _e('Delete Programs', 'agentic-press'); ?></h2>
                <p><?php _e('Permanently delete all affiliate links for selected programs. This action cannot be undone.', 'agentic-press'); ?></p>

                <form method="post" action="" onsubmit="return confirm('<?php _e('Are you sure you want to permanently delete all affiliate links for the selected programs? This action cannot be undone.', 'agentic-press'); ?>');">
                    <?php wp_nonce_field('delete_merchants_action', 'delete_merchants_nonce'); ?>

                    <div class="merchants-list-section">
                        <h3><?php _e('Select Programs to Delete', 'agentic-press'); ?></h3>
                        <div class="merchant-checkboxes">
                            <?php foreach ($merchants as $merchant): ?>
                                <label class="merchant-checkbox-item">
                                    <input type="checkbox"
                                           name="merchants_to_delete[]"
                                           value="<?php echo esc_attr($merchant['affiliate_program']); ?>" />
                                    <span class="merchant-info">
                                        <strong><?php echo esc_html($merchant['affiliate_program']); ?></strong>
                                        <span class="link-count">(<?php echo intval($merchant['link_count']); ?> links)</span>
                                    </span>
                                </label>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <div class="action-controls">
                        <button type="submit" name="delete_merchants" class="button button-danger">
                            <?php _e('Delete Selected Programs', 'agentic-press'); ?>
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <style>
            .affiliate-section {
                background: #fff;
                border: 1px solid #ccd0d4;
                border-radius: 4px;
                margin: 20px 0;
                padding: 20px;
            }

            .affiliate-delete-section {
                border-color: #dc3545;
                background: #fff5f5;
            }

            .affiliate-delete-section h2 {
                color: #dc3545;
            }

            .merchants-list-section {
                margin: 15px 0;
            }

            .merchant-checkboxes {
                display: grid;
                grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
                gap: 10px;
                margin: 15px 0;
                max-height: 400px;
                overflow-y: auto;
                border: 1px solid #e1e1e1;
                border-radius: 4px;
                padding: 15px;
                background: #f8f9fa;
            }

            .merchant-checkbox-item {
                display: flex;
                align-items: center;
                padding: 8px 12px;
                border: 1px solid #ddd;
                border-radius: 4px;
                background: #fff;
                cursor: pointer;
                transition: all 0.2s;
            }

            .merchant-checkbox-item:hover {
                border-color: #999;
                background: #f8f9fa;
            }

            .merchant-checkbox-item input[type="checkbox"] {
                margin-right: 10px;
                transform: scale(1.2);
            }

            .merchant-info {
                display: flex;
                flex-direction: column;
                flex: 1;
            }

            .merchant-info strong {
                font-size: 14px;
                color: #23282d;
            }

            .link-count {
                font-size: 12px;
                color: #666;
                font-weight: normal;
            }

            .action-controls {
                display: flex;
                align-items: center;
                gap: 10px;
                margin-top: 20px;
                padding-top: 15px;
                border-top: 1px solid #e1e1e1;
            }

            .button-danger {
                background: #dc3545;
                border-color: #dc3545;
                color: #fff;
            }

            .button-danger:hover {
                background: #c82333;
                border-color: #bd2130;
            }

            .affiliate-delete-section .merchant-checkboxes {
                background: #fef2f2;
                border-color: #fecaca;
            }

            .affiliate-delete-section .merchant-checkbox-item {
                border-color: #fecaca;
            }

            .affiliate-delete-section .merchant-checkbox-item:hover {
                background: #fef2f2;
                border-color: #fca5a5;
            }
        </style>
        <?php
    }

    /**
     * Get all affiliate merchants with their status
     */
    private function get_affiliate_merchants() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ap_affiliate_links';

        $sql = "SELECT DISTINCT
                    affiliate_program,
                    MIN(is_active) as is_active,
                    COUNT(*) as link_count
                FROM {$table_name}
                GROUP BY affiliate_program
                ORDER BY affiliate_program ASC";

        $results = $wpdb->get_results($sql, ARRAY_A);

        return $results ?: array();
    }

    /**
     * Handle update merchants action
     * Checked merchants = enable, unchecked merchants = disable
     */
    private function handle_update_merchants($checked_merchants) {
        if (!wp_verify_nonce($_POST['update_merchants_nonce'], 'update_merchants_action')) {
            wp_die(__('Security check failed', 'agentic-press'));
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'ap_affiliate_links';

        // Get all merchants from database
        $all_merchants = $this->get_affiliate_merchants();
        $all_merchant_programs = array_column($all_merchants, 'affiliate_program');

        // Convert checked merchants to array if it's not already
        if (!is_array($checked_merchants)) {
            $checked_merchants = array();
        }

        $enabled_count = 0;
        $disabled_count = 0;

        // Enable checked merchants
        if (!empty($checked_merchants)) {
            $placeholders = str_repeat('%s,', count($checked_merchants) - 1) . '%s';
            $sql = $wpdb->prepare(
                "UPDATE {$table_name} SET is_active = 1 WHERE affiliate_program IN ({$placeholders})",
                $checked_merchants
            );
            $enabled_count = $wpdb->query($sql);
        }

        // Disable unchecked merchants
        $unchecked_merchants = array_diff($all_merchant_programs, $checked_merchants);
        if (!empty($unchecked_merchants)) {
            $placeholders = str_repeat('%s,', count($unchecked_merchants) - 1) . '%s';
            $sql = $wpdb->prepare(
                "UPDATE {$table_name} SET is_active = 0 WHERE affiliate_program IN ({$placeholders})",
                $unchecked_merchants
            );
            $disabled_count = $wpdb->query($sql);
        }

        $message = sprintf(
            __('Status updated: %d links enabled, %d links disabled.', 'agentic-press'),
            $enabled_count,
            $disabled_count
        );

        add_settings_error(
            'affiliate_management',
            'status_updated',
            $message,
            'success'
        );
    }

    /**
     * Handle delete merchants action
     */
    private function handle_delete_merchants($merchants) {
        if (!wp_verify_nonce($_POST['delete_merchants_nonce'], 'delete_merchants_action')) {
            wp_die(__('Security check failed', 'agentic-press'));
        }

        if (empty($merchants)) {
            add_settings_error(
                'affiliate_management',
                'no_merchants_selected',
                __('Please select at least one program to delete.', 'agentic-press'),
                'error'
            );
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'ap_affiliate_links';

        $merchant_placeholders = str_repeat('%s,', count($merchants) - 1) . '%s';

        $sql = $wpdb->prepare(
            "DELETE FROM {$table_name} WHERE affiliate_program IN ({$merchant_placeholders})",
            $merchants
        );
        $affected = $wpdb->query($sql);

        $message = sprintf(
            __('Successfully deleted %d affiliate links for %d program(s).', 'agentic-press'),
            $affected,
            count($merchants)
        );

        add_settings_error(
            'affiliate_management',
            'delete_completed',
            $message,
            'success'
        );
    }
}
