<?php
/**
 * API endpoints for Author Category Ownership plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class AP_API {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('rest_api_init', array($this, 'register_routes'));
    }

    /**
     * Register REST API routes
     */
    public function register_routes() {
        register_rest_route('ap/v1', '/authors', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_authors'),
            'permission_callback' => array($this, 'check_permissions'),
            'args' => array(
                'category' => array(
                    'required' => false,
                    'validate_callback' => function($param) {
                        return is_string($param);
                    }
                ),
                'include_subcategories' => array(
                    'required' => false,
                    'default' => false,
                    'validate_callback' => function($param) {
                        return is_bool($param) || $param === 'true' || $param === 'false';
                    },
                    'sanitize_callback' => function($param) {
                        return filter_var($param, FILTER_VALIDATE_BOOLEAN);
                    }
                )
            )
        ));

        register_rest_route('ap/v1', '/categories', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_categories'),
            'permission_callback' => array($this, 'check_permissions'),
            'args' => array(
                'active_only' => array(
                    'required' => false,
                    'default' => true,
                    'validate_callback' => function($param) {
                        return is_bool($param) || $param === 'true' || $param === 'false';
                    },
                    'sanitize_callback' => function($param) {
                        return filter_var($param, FILTER_VALIDATE_BOOLEAN);
                    }
                )
            )
        ));
    }

    /**
     * Check API permissions
     */
    public function check_permissions() {
        // Allow authenticated users for now; adjust as needed
        return current_user_can('read');
    }

    /**
     * Get authors endpoint
     */
    public function get_authors($request) {
        $category_slug = $request->get_param('category');
        $include_subcategories = $request->get_param('include_subcategories');

        if (!$category_slug) {
            return new WP_Error('missing_category', 'Category parameter is required', array('status' => 400));
        }

        $authors = AP_DB::get_authors_by_category($category_slug, $include_subcategories);

        $response = array();
        foreach ($authors as $author) {
            $category = AP_DB::get_category_by_author($author->ID);
            $response[] = array(
                'id' => $author->ID,
                'name' => $author->display_name,
                'email' => $author->user_email,
                'category' => $category ? array(
                    'id' => $category->id,
                    'slug' => $category->category_slug,
                    'name' => $category->name,
                    'description' => $category->description,
                    'system_prompt' => $category->system_prompt,
                    'user_prompt' => $category->user_prompt,
                    'llm_provider' => $category->llm_provider,
                    'model' => $category->model,
                    'temperature' => $category->temperature,
                    'top_p' => $category->top_p,
                    'max_tokens' => $category->max_tokens,
                    'include_base_system_prompt' => (bool) $category->include_base_system_prompt
                ) : null
            );
        }

        return new WP_REST_Response($response, 200);
    }

    /**
     * Get categories endpoint
     */
    public function get_categories($request) {
        $active_only = $request->get_param('active_only');

        $categories = AP_DB::get_all_categories($active_only);

        $response = array();
        foreach ($categories as $category) {
            $response[] = array(
                'id' => $category->id,
                'slug' => $category->category_slug,
                'name' => $category->name,
                'description' => $category->description,
                'system_prompt' => $category->system_prompt,
                'user_prompt' => $category->user_prompt,
                'llm_provider' => $category->llm_provider,
                'model' => $category->model,
                'temperature' => $category->temperature,
                'top_p' => $category->top_p,
                'max_tokens' => $category->max_tokens,
                'include_base_system_prompt' => (bool) $category->include_base_system_prompt,
                'parent_id' => $category->parent_id,
                'is_active' => (bool) $category->is_active
            );
        }

        return new WP_REST_Response($response, 200);
    }
}