<?php
/**
 * Database operations for Author Category Ownership plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class AP_DB {

    /**
     * Create database tables
     */
    public static function create_tables() {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        // Categories table
        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;
        $sql_categories = "CREATE TABLE $table_categories (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            term_id bigint(20) unsigned NOT NULL,
            category_slug varchar(100) NOT NULL UNIQUE,
            name varchar(255) NOT NULL,
            description text,
            system_prompt text,
            user_prompt text,
            llm_provider varchar(50) DEFAULT 'openai',
            model varchar(50) DEFAULT 'o4-mini',
            temperature decimal(3,2) DEFAULT 0.5,
            top_p decimal(3,2) DEFAULT 0.8,
            max_tokens int(11) DEFAULT 10000,
            include_base_system_prompt boolean DEFAULT true,
            parent_id mediumint(9) DEFAULT NULL,
            is_active boolean DEFAULT true,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY term_id (term_id),
            KEY parent_id (parent_id)
        ) $charset_collate;";

        // Authors-Categories relationship table
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';
        $sql_authors_categories = "CREATE TABLE $table_authors_categories (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            category_id mediumint(9) NOT NULL,
            author_id bigint(20) unsigned NOT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY unique_author_category (author_id, category_id),
            KEY category_id (category_id),
            KEY author_id (author_id)
        ) $charset_collate;";

        // API Users table
        $table_ap_users = $wpdb->prefix . 'ap_users';
        $sql_ap_users = "CREATE TABLE $table_ap_users (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            username varchar(255) NOT NULL UNIQUE,
            password_hash varchar(255) NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";

        // API Tokens table
        $table_ap_tokens = $wpdb->prefix . 'ap_tokens';
        $sql_ap_tokens = "CREATE TABLE $table_ap_tokens (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            token varchar(255) NOT NULL UNIQUE,
            user_id mediumint(9) NOT NULL,
            expires_at datetime DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY user_id (user_id),
            KEY expires_at (expires_at)
        ) $charset_collate;";

        // Affiliate Links table
        $table_affiliate_links = $wpdb->prefix . 'ap_affiliate_links';
        $sql_affiliate_links = "CREATE TABLE $table_affiliate_links (
            id int(11) NOT NULL AUTO_INCREMENT,
            post_id int(11) NOT NULL,
            affiliate_program varchar(255) NOT NULL,
            merchant_domain varchar(255) NOT NULL,
            product_name varchar(500) NOT NULL,
            product_url text NOT NULL,
            affiliate_url text NOT NULL,
            product_description text,
            is_active tinyint(1) DEFAULT 1,
            created_at timestamp DEFAULT CURRENT_TIMESTAMP,
            updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_post_id (post_id),
            KEY idx_affiliate_program (affiliate_program),
            KEY idx_merchant_domain (merchant_domain),
            KEY idx_is_active (is_active)
        ) $charset_collate;";

        // Rename old tables if they exist
        self::rename_old_tables();

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql_categories);
        dbDelta($sql_authors_categories);
        dbDelta($sql_ap_users);
        dbDelta($sql_ap_tokens);
        dbDelta($sql_affiliate_links);

        // Verify tables were created
        $tables_created = true;
        if (!$wpdb->get_var("SHOW TABLES LIKE '$table_categories'")) {
            error_log("AP: Failed to create categories table: $table_categories");
            $tables_created = false;
        }
        if (!$wpdb->get_var("SHOW TABLES LIKE '$table_authors_categories'")) {
            error_log("AP: Failed to create authors_categories table: $table_authors_categories");
            $tables_created = false;
        }
        if (!$wpdb->get_var("SHOW TABLES LIKE '$table_ap_users'")) {
            error_log("AP: Failed to create ap_users table: $table_ap_users");
            $tables_created = false;
        }
        if (!$wpdb->get_var("SHOW TABLES LIKE '$table_ap_tokens'")) {
            error_log("AP: Failed to create ap_tokens table: $table_ap_tokens");
            $tables_created = false;
        }
        if (!$wpdb->get_var("SHOW TABLES LIKE '$table_affiliate_links'")) {
            error_log("AP: Failed to create affiliate_links table: $table_affiliate_links");
            $tables_created = false;
        }

        if ($tables_created) {
            error_log("AP: All tables created successfully");
        } else {
            error_log("AP: Some tables failed to create");
        }

        // Update table to remove unique constraint if it exists
        self::update_authors_categories_table();
    }

    /**
     * Rename old tables to new names for consistency
     */
    public static function rename_old_tables() {
        global $wpdb;

        $old_tables = [
            $wpdb->prefix . 'aco_categories' => $wpdb->prefix . AP_TABLE_CATEGORIES,
            $wpdb->prefix . 'aco_authors_categories' => $wpdb->prefix . AP_TABLE_AUTHORS_CATEGORIES,
            $wpdb->prefix . 'affiliate_links' => $wpdb->prefix . 'ap_affiliate_links'
        ];

        foreach ($old_tables as $old_table => $new_table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '$old_table'") && !$wpdb->get_var("SHOW TABLES LIKE '$new_table'")) {
                $wpdb->query("ALTER TABLE $old_table RENAME TO $new_table");
                error_log("AP: Renamed table $old_table to $new_table");
            }
        }
    }

    /**
     * Update authors-categories table to allow multiple assignments
     */
    public static function update_authors_categories_table() {
        global $wpdb;
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';

        // Remove duplicate entries
        $wpdb->query("
            DELETE t1 FROM $table_authors_categories t1
            INNER JOIN $table_authors_categories t2 
            WHERE t1.id > t2.id 
            AND t1.author_id = t2.author_id 
            AND t1.category_id = t2.category_id
        ");

        // Check if old unique index exists and drop it
        $indexes = $wpdb->get_results("SHOW INDEX FROM $table_authors_categories WHERE Key_name = 'unique_author_category'");
        if (!empty($indexes)) {
            $wpdb->query("ALTER TABLE $table_authors_categories DROP INDEX unique_author_category");
        }

        // Add new unique index on (author_id, category_id)
        $indexes = $wpdb->get_results("SHOW INDEX FROM $table_authors_categories WHERE Key_name = 'unique_author_category'");
        if (empty($indexes)) {
            $wpdb->query("ALTER TABLE $table_authors_categories ADD UNIQUE KEY unique_author_category (author_id, category_id)");
        }
    }

    /**
     * Migrate categories from JSON file
     */
    public static function migrate_categories_json() {
        $json_file = AP_PLUGIN_DIR . 'categories.json'; // Path to categories.json in plugin directory
        $log_file = AP_PLUGIN_DIR . 'migration.log'; // Log file in plugin directory

        if (!file_exists($json_file)) {
            file_put_contents($log_file, 'ACO Migration: categories.json not found at ' . $json_file . "\n", FILE_APPEND);
            return;
        }

        $json_content = file_get_contents($json_file);
        if ($json_content === false) {
            file_put_contents($log_file, 'ACO Migration: Failed to read categories.json' . "\n", FILE_APPEND);
            return;
        }

        $categories = json_decode($json_content, true);
        if (!$categories) {
            file_put_contents($log_file, 'ACO Migration: Failed to decode JSON. JSON error: ' . json_last_error_msg() . "\n", FILE_APPEND);
            return;
        }

        file_put_contents($log_file, 'AP Migration: Found ' . count($categories) . ' categories to migrate' . "\n", FILE_APPEND);

        global $wpdb;
        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';

        foreach ($categories as $category) {
            // Insert main category
            $result = $wpdb->insert(
                $table_categories,
                array(
                    'category_slug' => $category['category'],
                    'name' => $category['name'],
                    'description' => $category['description'],
                    'system_prompt' => $category['system_prompt'],
                    'user_prompt' => $category['user_prompt'],
                    'llm_provider' => $category['llm_provider'],
                    'model' => $category['model'],
                    'temperature' => $category['temperature'],
                    'top_p' => $category['top_p'],
                    'max_tokens' => $category['max_tokens'],
                    'include_base_system_prompt' => $category['include_base_system_prompt'],
                    'parent_id' => null,
                    'is_active' => ($category['category'] !== 'disabled-architecture')
                )
            );

            if ($result === false) {
                file_put_contents($log_file, 'ACO Migration: Failed to insert category ' . $category['category'] . '. DB Error: ' . $wpdb->last_error . "\n", FILE_APPEND);
                continue;
            }

            $category_id = $wpdb->insert_id;
            file_put_contents($log_file, 'ACO Migration: Inserted category ' . $category['category'] . ' with ID ' . $category_id . "\n", FILE_APPEND);

            // Insert authors for this category
            if (isset($category['authors']['envs'])) {
                foreach ($category['authors']['envs'] as $env) {
                    if (isset($env['authors'])) {
                        foreach ($env['authors'] as $author) {
                            $user = get_user_by('id', $author['id']);
                            if ($user) {
                                $wpdb->insert(
                                    $table_authors_categories,
                                    array(
                                        'category_id' => $category_id,
                                        'author_id' => $author['id']
                                    ),
                                    array('%d', '%d')
                                );
                            }
                        }
                    }
                }
            }

            // Insert subcategories
            if (isset($category['sub_categories']) && is_array($category['sub_categories'])) {
                self::insert_subcategories($category['sub_categories'], $category_id);
            }
        }
    }

    /**
     * Migrate categories from JSON array (for import functionality)
     */
    public static function migrate_categories_json_from_array($categories) {
        $log_file = AP_PLUGIN_DIR . 'migration.log'; // Log file in plugin directory

        if (!is_array($categories)) {
            file_put_contents($log_file, 'ACO Import: Invalid categories array' . "\n", FILE_APPEND);
            return;
        }

        file_put_contents($log_file, 'AP Import: Found ' . count($categories) . ' categories to import' . "\n", FILE_APPEND);

        global $wpdb;
        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';

        // Clear existing categories and assignments
        $wpdb->query("DELETE FROM $table_authors_categories");
        $wpdb->query("DELETE FROM $table_categories");

        foreach ($categories as $category) {
            // Insert main category
            $result = $wpdb->insert(
                $table_categories,
                array(
                    'category_slug' => $category['category'],
                    'name' => $category['name'],
                    'description' => $category['description'],
                    'system_prompt' => $category['system_prompt'],
                    'user_prompt' => $category['user_prompt'],
                    'llm_provider' => $category['llm_provider'],
                    'model' => $category['model'],
                    'temperature' => $category['temperature'],
                    'top_p' => $category['top_p'],
                    'max_tokens' => $category['max_tokens'],
                    'include_base_system_prompt' => $category['include_base_system_prompt'],
                    'parent_id' => null,
                    'is_active' => true
                )
            );

            if ($result === false) {
                file_put_contents($log_file, 'ACO Import: Failed to insert category ' . $category['category'] . '. DB Error: ' . $wpdb->last_error . "\n", FILE_APPEND);
                continue;
            }

            $category_id = $wpdb->insert_id;
            file_put_contents($log_file, 'ACO Import: Inserted category ' . $category['category'] . ' with ID ' . $category_id . "\n", FILE_APPEND);

            // Insert authors for this category
            if (isset($category['authors']['envs'])) {
                foreach ($category['authors']['envs'] as $env) {
                    if (isset($env['authors'])) {
                        foreach ($env['authors'] as $author) {
                            $user = get_user_by('id', $author['id']);
                            if ($user) {
                                $wpdb->insert(
                                    $table_authors_categories,
                                    array(
                                        'category_id' => $category_id,
                                        'author_id' => $author['id']
                                    ),
                                    array('%d', '%d')
                                );
                            } else {
                                file_put_contents($log_file, 'ACO Import: Author ID ' . $author['id'] . ' not found, skipping assignment' . "\n", FILE_APPEND);
                            }
                        }
                    }
                }
            }

            // Insert subcategories
            if (isset($category['sub_categories']) && is_array($category['sub_categories'])) {
                self::insert_subcategories($category['sub_categories'], $category_id);
            }
        }

        file_put_contents($log_file, 'AP Import: Import completed' . "\n", FILE_APPEND);
    }

    /**
     * Insert subcategories recursively
     */
    private static function insert_subcategories($subcategories, $parent_id) {
        global $wpdb;
        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;

        foreach ($subcategories as $subcategory) {
            $wpdb->insert(
                $table_categories,
                array(
                    'category_slug' => $subcategory['category'],
                    'name' => $subcategory['name'],
                    'description' => $subcategory['description'],
                    'system_prompt' => isset($subcategory['system_prompt']) ? $subcategory['system_prompt'] : null,
                    'user_prompt' => isset($subcategory['user_prompt']) ? $subcategory['user_prompt'] : null,
                    'llm_provider' => isset($subcategory['llm_provider']) ? $subcategory['llm_provider'] : 'openai',
                    'model' => isset($subcategory['model']) ? $subcategory['model'] : 'o4-mini',
                    'temperature' => isset($subcategory['temperature']) ? $subcategory['temperature'] : 0.5,
                    'top_p' => isset($subcategory['top_p']) ? $subcategory['top_p'] : 0.8,
                    'max_tokens' => isset($subcategory['max_tokens']) ? $subcategory['max_tokens'] : 10000,
                    'include_base_system_prompt' => isset($subcategory['include_base_system_prompt']) ? $subcategory['include_base_system_prompt'] : true,
                    'parent_id' => $parent_id,
                    'is_active' => true
                )
            );

            $subcategory_id = $wpdb->insert_id;

            // Recursively insert deeper subcategories
            if (isset($subcategory['sub_categories']) && is_array($subcategory['sub_categories'])) {
                self::insert_subcategories($subcategory['sub_categories'], $subcategory_id);
            }
        }
    }

    /**
     * Get authors by category
     */
    public static function get_authors_by_category($category_slug, $include_subcategories = false) {
        global $wpdb;

        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';

        $category_ids = array();

        // Get main category ID
        $category = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $table_categories WHERE category_slug = %s AND is_active = 1",
            $category_slug
        ));

        if (!$category) {
            return array();
        }

        $category_ids[] = $category->id;

        // Include subcategories if requested
        if ($include_subcategories) {
            $subcategories = self::get_subcategories($category->id);
            $category_ids = array_merge($category_ids, $subcategories);
        }

        // Get authors
        $placeholders = implode(',', array_fill(0, count($category_ids), '%d'));
        $authors = $wpdb->get_results($wpdb->prepare(
            "SELECT u.ID, u.display_name, u.user_email, ac.category_id
             FROM {$wpdb->users} u
             INNER JOIN $table_authors_categories ac ON u.ID = ac.author_id
             WHERE ac.category_id IN ($placeholders)
             ORDER BY u.display_name",
            $category_ids
        ));

        return $authors;
    }

    /**
     * Get subcategories recursively
     */
    private static function get_subcategories($parent_id) {
        global $wpdb;
        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;

        $subcategory_ids = array();
        $subcategories = $wpdb->get_results($wpdb->prepare(
            "SELECT id FROM $table_categories WHERE parent_id = %d AND is_active = 1",
            $parent_id
        ));

        foreach ($subcategories as $subcategory) {
            $subcategory_ids[] = $subcategory->id;
            // Recursively get deeper subcategories
            $deeper = self::get_subcategories($subcategory->id);
            $subcategory_ids = array_merge($subcategory_ids, $deeper);
        }

        return $subcategory_ids;
    }

    /**
     * Get category by slug
     */
    public static function get_category_by_slug($category_slug) {
        global $wpdb;

        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;

        $category = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_categories WHERE category_slug = %s",
            $category_slug
        ));

        return $category;
    }

    /**
     * Get categories by author ID
     */
    public static function get_categories_by_author($author_id) {
        global $wpdb;

        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';

        $categories = $wpdb->get_results($wpdb->prepare(
            "SELECT DISTINCT c.* FROM $table_categories c
             INNER JOIN $table_authors_categories ac ON c.id = ac.category_id
             WHERE ac.author_id = %d AND c.is_active = 1
             ORDER BY c.name",
            $author_id
        ));

        return $categories;
    }

    /**
     * Get category by author ID (deprecated - use get_categories_by_author)
     */
    public static function get_category_by_author($author_id) {
        $categories = self::get_categories_by_author($author_id);
        return !empty($categories) ? $categories[0] : null;
    }

    /**
     * Get category by ID
     */
    public static function get_category_by_id($category_id) {
        global $wpdb;

        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;

        $category = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_categories WHERE id = %d",
            $category_id
        ));

        return $category;
    }

    /**
     * Get all categories
     */
    public static function get_all_categories($active_only = true) {
        global $wpdb;

        $table_categories = $wpdb->prefix . AP_TABLE_CATEGORIES;

        $where = $active_only ? 'WHERE is_active = 1' : '';

        $categories = $wpdb->get_results(
            "SELECT * FROM $table_categories $where ORDER BY name ASC"
        );

        return $categories;
    }

    /**
     * Assign category to author
     */
    public static function assign_category_to_author($author_id, $category_id) {
        global $wpdb;
        $table_authors_categories = $wpdb->prefix . 'ap_authors_categories';

        // Remove existing assignment
        $wpdb->delete($table_authors_categories, array('author_id' => $author_id), array('%d'));

        // Add new assignment
        if ($category_id > 0) {
            $wpdb->insert(
                $table_authors_categories,
                array('category_id' => $category_id, 'author_id' => $author_id),
                array('%d', '%d')
            );
        }
    }}
