<?php
/**
 * Agentic Press Frontend Functionality
 *
 * Handles shortcodes, scripts, and template parts that were previously in the child theme
 */

if (!defined('ABSPATH')) {
    exit;
}

class AP_Frontend {

    public function __construct() {
        $this->init_hooks();
    }

    private function init_hooks() {
        // Enqueue frontend styles - only on frontend, not admin
        add_action('wp_enqueue_scripts', array($this, 'enqueue_styles'));

        // Add affiliate disclosure
        add_action('blocksy:single:content:top', array($this, 'add_affiliate_disclosure'));

        // Register shortcodes
        add_shortcode('affiliate_box', array($this, 'affiliate_box_shortcode'));
    }

    /**
     * Add affiliate disclosure to posts with affiliate tag
     */
    public function add_affiliate_disclosure() {
        if (is_single() && has_tag('affiliate')) {
            $this->render_affiliate_disclosure();
        }
    }

    /**
     * Render affiliate disclosure HTML
     */
    private function render_affiliate_disclosure() {
        ?>
        <div class="affiliate-disclosure">
            <em>This post contains affiliate links. If you buy something, I may earn a small commission—no extra cost to you, and it helps keep the lights on.</em>
        </div>
        <?php
    }

    /**
     * Enqueue frontend styles
     */
    public function enqueue_styles() {
        // Only enqueue on frontend, not in admin
        if (!is_admin()) {
            wp_enqueue_style('agentic-press-frontend-css', AP_PLUGIN_URL . 'assets/css/style.css', array(), AP_VERSION);
        }
    }

    /**
     * Affiliate box shortcode handler
     */
    public function affiliate_box_shortcode($atts) {
        $atts = shortcode_atts(array('post_id' => 0), $atts, 'affiliate_box');
        $post_id = intval($atts['post_id']);

        // If no post_id provided, try to get current post ID
        if ($post_id <= 0) {
            global $post;
            if (isset($post) && isset($post->ID)) {
                $post_id = $post->ID;
            } else {
                return '<p>Affiliate Box: No post_id provided and could not determine current post ID</p>';
            }
        }

        if ($post_id <= 0) {
            return '<p>Affiliate Box: Invalid post_id: ' . $post_id . '</p>';
        }

        $affiliate_links = $this->get_affiliate_links_for_post($post_id);

        if (empty($affiliate_links)) {
            return '<p>No affiliate links found for this post.</p>';
        }

        // Check if we got an error response
        if (isset($affiliate_links['error'])) {
            return '<p>Affiliate Box Error: ' . esc_html($affiliate_links['error']) . '</p>';
        }

        $output = "\n\n<div class=\"affiliate-links-container\">\n";
        $output .= "<div class=\"affiliate-links-title\">\n";
        $output .= "<ul class=\"affiliate-links-list\">\n";

        foreach ($affiliate_links as $link) {
            $product_name = esc_html($link['product_name']);
            $affiliate_url = esc_url($link['affiliate_url']);
            $product_description = esc_html($link['product_description']);

            $output .= "<li class=\"affiliate-link-item\">\n";
            $output .= "    <a href=\"{$affiliate_url}\" target=\"_blank\" rel=\"nofollow sponsored\" class=\"affiliate-link\">\n";
            $output .= "        <strong>{$product_name}</strong>\n";
            if (!empty($product_description)) {
                $output .= "        <br><small>{$product_description}</small>\n";
            }
            $output .= "    </a>\n";
            $output .= "</li>\n";
        }

        $output .= "</ul>\n";
        $output .= "</div>\n\n";

        return $output;
    }

    /**
     * Get affiliate links for a post from database
     */
    private function get_affiliate_links_for_post($post_id) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'ap_affiliate_links';

        $links = $wpdb->get_results( $wpdb->prepare(
            "SELECT product_name, affiliate_url, product_description FROM {$table_name} WHERE post_id = %d AND is_active = 1 ORDER BY created_at",
            $post_id
        ) );

        if (!is_array($links)) {
            return array('error' => 'Database query failed');
        }

        // Convert to the expected format
        $result = array();
        foreach ($links as $link) {
            $result[] = array(
                'product_name' => $link->product_name,
                'affiliate_url' => $link->affiliate_url,
                'product_description' => $link->product_description,
            );
        }

        return $result;
    }
}
