<?php
/**
 * Agentic Press Settings Page
 *
 * Handles plugin configuration for SaaS service integration
 */

if (!defined('ABSPATH')) {
    exit;
}

class AP_Settings {

    private $option_name = 'agentic_press_settings';
    private $settings = array();

    public function __construct() {
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }

    /**
     * Get default settings
     */
    private function get_defaults() {
        return array(
            // Service Connection
            'service_username' => '',
            'service_password' => '',
            'service_site_url' => get_site_url(),
            'service_api_url' => '', // SaaS API endpoint URL

            // API Authentication (for SaaS to access WordPress)
            'api_auth_username' => '',
            'api_auth_password' => '',

            // AI Configuration
            'llm_provider' => 'openai',
            'openai_api_key' => '',
            'xai_api_key' => '',

            // Content Preferences
            'lookback_days' => 14,
            'enable_subcategory_labeling' => true,
            'max_subcategory_labels' => 2,

            // Media Settings
            'image_budget' => 'cost',
            'image_quality' => 'medium',
            'video_budget' => 'cost',
            'video_quality' => 'low',

            // Affiliate Settings
            'enable_affiliate_insights' => true,
            'max_affiliate_search_queries' => 2,
            'results_per_product' => 1,

            // Search API Settings
            'google_search_key' => '',
            'google_search_cx' => '',
            'serpapi_key' => '',
            'brave_search_key' => '',

            // Advanced
            'debug_mode' => false,
        );
    }

    /**
     * Get settings with defaults
     */
    public function get_settings() {
        if (empty($this->settings)) {
            $this->settings = wp_parse_args(
                get_option($this->option_name, array()),
                $this->get_defaults()
            );
        }
        return $this->settings;
    }

    /**
     * Get the API endpoint URL with fallback logic
     */
    public function get_api_url() {
        $settings = $this->get_settings();

        // User must configure a custom API URL since environment is determined by the URL
        if (!empty($settings['service_api_url'])) {
            return rtrim($settings['service_api_url'], '/');
        }

        // No fallback - user must specify the API URL
        return '';
    }

    /**
     * Register settings with WordPress
     */
    public function register_settings() {
        register_setting(
            $this->option_name,
            $this->option_name,
            array($this, 'sanitize_settings')
        );

        // Service Connection Section
        add_settings_section(
            'autopress_service',
            __('Service Connection', 'autopress'),
            array($this, 'service_section_callback'),
            'autopress-settings'
        );

        add_settings_field(
            'service_credentials',
            __('Agentic Press Credentials', 'autopress'),
            array($this, 'service_credentials_callback'),
            'autopress-settings',
            'autopress_service'
        );

        add_settings_field(
            'service_site_url',
            __('Site URL', 'autopress'),
            array($this, 'service_site_url_callback'),
            'autopress-settings',
            'autopress_service'
        );

        add_settings_field(
            'service_api_url',
            __('API Endpoint URL', 'autopress'),
            array($this, 'service_api_url_callback'),
            'autopress-settings',
            'autopress_service'
        );

        // API Authentication Section
        add_settings_section(
            'autopress_api_auth',
            __('API Authentication', 'autopress'),
            array($this, 'api_auth_section_callback'),
            'autopress-settings'
        );

        add_settings_field(
            'api_auth_credentials',
            __('API Credentials', 'autopress'),
            array($this, 'api_auth_credentials_callback'),
            'autopress-settings',
            'autopress_api_auth'
        );

        // AI Configuration Section
        add_settings_section(
            'autopress_ai',
            __('AI Configuration', 'autopress'),
            array($this, 'ai_section_callback'),
            'autopress-settings'
        );

        add_settings_field(
            'llm_provider',
            __('LLM Provider', 'autopress'),
            array($this, 'llm_provider_callback'),
            'autopress-settings',
            'autopress_ai'
        );

        add_settings_field(
            'api_keys',
            __('API Keys', 'autopress'),
            array($this, 'api_keys_callback'),
            'autopress-settings',
            'autopress_ai'
        );

        // Content Preferences Section
        add_settings_section(
            'autopress_content',
            __('Content Preferences', 'autopress'),
            array($this, 'content_section_callback'),
            'autopress-settings'
        );

        add_settings_field(
            'lookback_days',
            __('Lookback Days', 'autopress'),
            array($this, 'lookback_days_callback'),
            'autopress-settings',
            'autopress_content'
        );

        add_settings_field(
            'subcategory_settings',
            __('Subcategory Settings', 'autopress'),
            array($this, 'subcategory_settings_callback'),
            'autopress-settings',
            'autopress_content'
        );

        // Media Settings Section
        add_settings_section(
            'autopress_media',
            __('Media Settings', 'autopress'),
            array($this, 'media_section_callback'),
            'autopress-settings'
        );

        add_settings_field(
            'image_settings',
            __('Image Generation', 'autopress'),
            array($this, 'image_settings_callback'),
            'autopress-settings',
            'autopress_media'
        );

        add_settings_field(
            'video_settings',
            __('Video Generation', 'autopress'),
            array($this, 'video_settings_callback'),
            'autopress-settings',
            'autopress_media'
        );

        // Affiliate Settings Section
        add_settings_section(
            'autopress_affiliate',
            __('Affiliate Settings', 'autopress'),
            array($this, 'affiliate_section_callback'),
            'autopress-settings'
        );

        add_settings_field(
            'affiliate_insights',
            __('Affiliate Insights', 'autopress'),
            array($this, 'affiliate_insights_callback'),
            'autopress-settings',
            'autopress_affiliate'
        );

        add_settings_field(
            'google_search_api',
            __('Google Search API', 'autopress'),
            array($this, 'google_search_api_callback'),
            'autopress-settings',
            'autopress_affiliate'
        );

        // Advanced Settings Section
        add_settings_section(
            'autopress_advanced',
            __('Advanced Settings', 'autopress'),
            array($this, 'advanced_section_callback'),
            'autopress-settings'
        );

        add_settings_field(
            'debug_mode',
            __('Debug Mode', 'autopress'),
            array($this, 'debug_mode_callback'),
            'autopress-settings',
            'autopress_advanced'
        );
    }

    /**
     * Sanitize settings input
     */
    public function sanitize_settings($input) {
        $sanitized = array();

        // Service Connection
        $sanitized['service_username'] = sanitize_text_field($input['service_username'] ?? '');
        $sanitized['service_password'] = $input['service_password'] ?? ''; // Don't sanitize passwords
        $sanitized['service_site_url'] = esc_url_raw($input['service_site_url'] ?? '');
        $sanitized['service_api_url'] = esc_url_raw($input['service_api_url'] ?? '');

        // API Authentication
        $sanitized['api_auth_username'] = sanitize_text_field($input['api_auth_username'] ?? '');
        $sanitized['api_auth_password'] = $input['api_auth_password'] ?? ''; // Don't sanitize passwords

        // AI Configuration
        $sanitized['llm_provider'] = in_array($input['llm_provider'] ?? '', ['openai', 'xai']) ? $input['llm_provider'] : 'openai';
        $sanitized['openai_api_key'] = sanitize_text_field($input['openai_api_key'] ?? '');
        $sanitized['xai_api_key'] = sanitize_text_field($input['xai_api_key'] ?? '');

        // Content Preferences
        $sanitized['lookback_days'] = absint($input['lookback_days'] ?? 14);
        $sanitized['enable_subcategory_labeling'] = isset($input['enable_subcategory_labeling']);
        $sanitized['max_subcategory_labels'] = absint($input['max_subcategory_labels'] ?? 2);

        // Media Settings
        $sanitized['image_budget'] = in_array($input['image_budget'] ?? '', ['cost', 'quality']) ? $input['image_budget'] : 'cost';
        $sanitized['image_quality'] = in_array($input['image_quality'] ?? '', ['low', 'medium', 'high']) ? $input['image_quality'] : 'medium';
        $sanitized['video_budget'] = in_array($input['video_budget'] ?? '', ['cost', 'quality']) ? $input['video_budget'] : 'cost';
        $sanitized['video_quality'] = in_array($input['video_quality'] ?? '', ['low', 'medium', 'high', 'ultra']) ? $input['video_quality'] : 'low';

        // Affiliate Settings
        $sanitized['enable_affiliate_insights'] = isset($input['enable_affiliate_insights']);
        $sanitized['max_affiliate_search_queries'] = absint($input['max_affiliate_search_queries'] ?? 2);
        $sanitized['results_per_product'] = absint($input['results_per_product'] ?? 1);

        // Search API Settings
        $sanitized['google_search_key'] = sanitize_text_field($input['google_search_key'] ?? '');
        $sanitized['google_search_cx'] = sanitize_text_field($input['google_search_cx'] ?? '');
        $sanitized['serpapi_key'] = sanitize_text_field($input['serpapi_key'] ?? '');
        $sanitized['brave_search_key'] = sanitize_text_field($input['brave_search_key'] ?? '');

        // Advanced
        $sanitized['debug_mode'] = isset($input['debug_mode']);

        return $sanitized;
    }

    /**
     * Get list of API users for dropdown
     */
    private function get_api_users() {
        global $wpdb;
        $table_users = $wpdb->prefix . 'ap_users';
        return $wpdb->get_results("SELECT username FROM $table_users ORDER BY username ASC");
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_scripts($hook) {
        if ($hook !== 'toplevel_page_agentic-press-main') {
            return;
        }

        wp_enqueue_script('autopress-settings', AP_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), AP_VERSION, true);
        wp_enqueue_style('autopress-admin', AP_PLUGIN_URL . 'assets/css/admin.css', array(), AP_VERSION);
    }

    // Section Callbacks
    public function service_section_callback() {
        echo '<p>' . __('Configure your connection to the Agentic Press service.', 'agentic-press') . '</p>';
    }

    public function api_auth_section_callback() {
        echo '<p>' . __('Configure API authentication credentials for the Agentic Press service to access your WordPress site.', 'agentic-press') . '</p>';
    }

    public function ai_section_callback() {
        echo '<p>' . __('Configure AI providers and API keys for content generation.', 'agentic-press') . '</p>';
    }

    public function content_section_callback() {
        echo '<p>' . __('Control how content is generated and published.', 'agentic-press') . '</p>';
    }

    public function media_section_callback() {
        echo '<p>' . __('Configure image and video generation settings.', 'agentic-press') . '</p>';
    }

    public function affiliate_section_callback() {
        echo '<p>' . __('Control affiliate product integration.', 'agentic-press') . '</p>';
    }

    public function advanced_section_callback() {
        echo '<p>' . __('Advanced settings for debugging and troubleshooting.', 'agentic-press') . '</p>';
    }

    // Field Callbacks
    public function service_credentials_callback() {
        $settings = $this->get_settings();
        ?>
        <table class="form-table">
            <tr>
                <th><label for="service_username"><?php _e('Username', 'agentic-press'); ?></label></th>
                <td>
                    <input type="text" id="service_username" name="<?php echo $this->option_name; ?>[service_username]"
                           value="<?php echo esc_attr($settings['service_username']); ?>" class="regular-text">
                    <p class="description"><?php _e('Username for Agentic Press service to access your site.', 'agentic-press'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="service_password"><?php _e('Password', 'agentic-press'); ?></label></th>
                <td>
                    <input type="password" id="service_password" name="<?php echo $this->option_name; ?>[service_password]"
                           value="<?php echo esc_attr($settings['service_password']); ?>" class="regular-text">
                    <button type="button" id="toggle_password" class="button button-small"><?php _e('Show', 'agentic-press'); ?></button>
                    <p class="description"><?php _e('Password for Agentic Press service to access your site.', 'agentic-press'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }

    public function service_site_url_callback() {
        $settings = $this->get_settings();
        ?>
        <input type="url" id="service_site_url" name="<?php echo $this->option_name; ?>[service_site_url]"
               value="<?php echo esc_attr($settings['service_site_url']); ?>" class="regular-text">
        <p class="description"><?php _e('Your WordPress site URL for the Agentic Press service to publish to.', 'agentic-press'); ?></p>
        <?php
    }

    public function service_api_url_callback() {
        $settings = $this->get_settings();
        ?>
        <input type="url" id="service_api_url" name="<?php echo $this->option_name; ?>[service_api_url]"
               value="<?php echo esc_attr($settings['service_api_url']); ?>" class="regular-text" placeholder="https://your-api-domain.com/api">
        <p class="description"><?php _e('The Agentic Press SaaS API endpoint URL. Leave blank to use the default service.', 'agentic-press'); ?></p>
        <?php
    }

    public function api_auth_credentials_callback() {
        $settings = $this->get_settings();
        $api_users = $this->get_api_users();
        ?>
        <table class="form-table">
            <tr>
                <th><label for="api_auth_username"><?php _e('API Username', 'autopress'); ?></label></th>
                <td>
                    <select id="api_auth_username" name="<?php echo $this->option_name; ?>[api_auth_username]" class="regular-text">
                        <option value=""><?php _e('-- Select API User --', 'agentic-press'); ?></option>
                        <?php foreach ($api_users as $user): ?>
                            <option value="<?php echo esc_attr($user->username); ?>" <?php selected($settings['api_auth_username'], $user->username); ?>>
                                <?php echo esc_html($user->username); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <p class="description">
                        <?php _e('Select an API user for the Agentic Press service to authenticate with your WordPress site.', 'agentic-press'); ?>
                        <a href="<?php echo admin_url('admin.php?page=agentic-press-api-auth'); ?>" target="_blank"><?php _e('Manage API Users', 'agentic-press'); ?></a>
                    </p>
                </td>
            </tr>
            <tr>
                <th><label for="api_auth_password"><?php _e('API Password', 'agentic-press'); ?></label></th>
                <td>
                    <input type="password" id="api_auth_password" name="<?php echo $this->option_name; ?>[api_auth_password]"
                           value="<?php echo esc_attr($settings['api_auth_password']); ?>" class="regular-text">
                    <button type="button" id="toggle_api_auth_password" class="button button-small"><?php _e('Show', 'agentic-press'); ?></button>
                    <p class="description">
                        <?php _e('Enter the password you set when creating this API user. Passwords are stored securely and cannot be retrieved - if forgotten, you\'ll need to reset it on the API Authentication page.', 'agentic-press'); ?>
                    </p>
                </td>
            </tr>
        </table>
        <script>
        document.getElementById('toggle_api_auth_password').addEventListener('click', function() {
            var passwordField = document.getElementById('api_auth_password');
            var button = this;
            
            if (passwordField.type === 'password') {
                passwordField.type = 'text';
                button.textContent = '<?php _e('Hide', 'agentic-press'); ?>';
            } else {
                passwordField.type = 'password';
                button.textContent = '<?php _e('Show', 'agentic-press'); ?>';
            }
        });

        // Clear password field when username selection changes to avoid confusion
        document.getElementById('api_auth_username').addEventListener('change', function() {
            var passwordField = document.getElementById('api_auth_password');
            if (passwordField.value) {
                passwordField.value = '';
                // Reset toggle button text if it's currently showing "Hide"
                var toggleButton = document.getElementById('toggle_api_auth_password');
                if (toggleButton.textContent === '<?php _e('Hide', 'autopress'); ?>') {
                    toggleButton.textContent = '<?php _e('Show', 'autopress'); ?>';
                    passwordField.type = 'password';
                }
            }
        });
        </script>
        <?php
    }

    public function llm_provider_callback() {
        $settings = $this->get_settings();
        $providers = array('openai' => 'OpenAI', 'xai' => 'xAI');
        ?>
        <select name="<?php echo $this->option_name; ?>[llm_provider]" id="llm_provider">
            <?php foreach ($providers as $value => $label): ?>
                <option value="<?php echo esc_attr($value); ?>" <?php selected($settings['llm_provider'], $value); ?>>
                    <?php echo esc_html($label); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <p class="description"><?php _e('Choose your preferred AI provider for content generation.', 'agentic-press'); ?></p>
        <?php
    }

    public function api_keys_callback() {
        $settings = $this->get_settings();
        ?>
        <table class="form-table">
            <tr>
                <th><label for="openai_api_key"><?php _e('OpenAI API Key', 'autopress'); ?></label></th>
                <td>
                    <input type="password" id="openai_api_key" name="<?php echo $this->option_name; ?>[openai_api_key]"
                           value="<?php echo esc_attr($settings['openai_api_key']); ?>" class="regular-text">
                    <p class="description"><?php _e('Your OpenAI API key (you pay for your own usage).', 'agentic-press'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="xai_api_key"><?php _e('xAI API Key', 'agentic-press'); ?></label></th>
                <td>
                    <input type="password" id="xai_api_key" name="<?php echo $this->option_name; ?>[xai_api_key]"
                           value="<?php echo esc_attr($settings['xai_api_key']); ?>" class="regular-text">
                    <p class="description"><?php _e('Your xAI API key (you pay for your own usage).', 'agentic-press'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }

    public function lookback_days_callback() {
        $settings = $this->get_settings();
        ?>
        <input type="number" id="lookback_days" name="<?php echo $this->option_name; ?>[lookback_days]"
               value="<?php echo esc_attr($settings['lookback_days']); ?>" min="1" max="365" class="small-text">
        <p class="description"><?php _e('Days to look back for recent posts to avoid duplicates.', 'agentic-press'); ?></p>
        <?php
    }

    public function subcategory_settings_callback() {
        $settings = $this->get_settings();
        ?>
        <table class="form-table">
            <tr>
                <th><label for="enable_subcategory_labeling"><?php _e('Enable Subcategory Labeling', 'autopress'); ?></label></th>
                <td>
                    <input type="checkbox" id="enable_subcategory_labeling" name="<?php echo $this->option_name; ?>[enable_subcategory_labeling]"
                           <?php checked($settings['enable_subcategory_labeling']); ?>>
                    <p class="description"><?php _e('Allow AI to assign subcategories to posts.', 'agentic-press'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="max_subcategory_labels"><?php _e('Max Subcategory Labels', 'agentic-press'); ?></label></th>
                <td>
                    <input type="number" id="max_subcategory_labels" name="<?php echo $this->option_name; ?>[max_subcategory_labels]"
                           value="<?php echo esc_attr($settings['max_subcategory_labels']); ?>" min="1" max="10" class="small-text">
                    <p class="description"><?php _e('Maximum number of subcategories per post.', 'agentic-press'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }

    public function image_settings_callback() {
        $settings = $this->get_settings();
        ?>
        <table class="form-table">
            <tr>
                <th><label for="image_budget"><?php _e('Image Budget', 'agentic-press'); ?></label></th>
                <td>
                    <select name="<?php echo $this->option_name; ?>[image_budget]" id="image_budget">
                        <option value="cost" <?php selected($settings['image_budget'], 'cost'); ?>><?php _e('Cost Optimized', 'agentic-press'); ?></option>
                        <option value="quality" <?php selected($settings['image_budget'], 'quality'); ?>><?php _e('Quality Optimized', 'agentic-press'); ?></option>
                    </select>
                    <p class="description"><?php _e('Balance between cost and quality for image generation.', 'agentic-press'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="image_quality"><?php _e('Image Quality', 'agentic-press'); ?></label></th>
                <td>
                    <select name="<?php echo $this->option_name; ?>[image_quality]" id="image_quality">
                        <option value="low" <?php selected($settings['image_quality'], 'low'); ?>><?php _e('Low', 'agentic-press'); ?></option>
                        <option value="medium" <?php selected($settings['image_quality'], 'medium'); ?>><?php _e('Medium', 'agentic-press'); ?></option>
                        <option value="high" <?php selected($settings['image_quality'], 'high'); ?>><?php _e('High', 'agentic-press'); ?></option>
                    </select>
                    <p class="description"><?php _e('Quality level for generated images.', 'autopress'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }

    public function video_settings_callback() {
        $settings = $this->get_settings();
        ?>
        <table class="form-table">
            <tr>
                <th><label for="video_budget"><?php _e('Video Budget', 'agentic-press'); ?></label></th>
                <td>
                    <select name="<?php echo $this->option_name; ?>[video_budget]" id="video_budget">
                        <option value="cost" <?php selected($settings['video_budget'], 'cost'); ?>><?php _e('Cost Optimized', 'agentic-press'); ?></option>
                        <option value="quality" <?php selected($settings['video_budget'], 'quality'); ?>><?php _e('Quality Optimized', 'agentic-press'); ?></option>
                    </select>
                    <p class="description"><?php _e('Balance between cost and quality for video generation.', 'agentic-press'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="video_quality"><?php _e('Video Quality', 'agentic-press'); ?></label></th>
                <td>
                    <select name="<?php echo $this->option_name; ?>[video_quality]" id="video_quality">
                        <option value="low" <?php selected($settings['video_quality'], 'low'); ?>><?php _e('Low', 'agentic-press'); ?></option>
                        <option value="medium" <?php selected($settings['video_quality'], 'medium'); ?>><?php _e('Medium', 'agentic-press'); ?></option>
                        <option value="high" <?php selected($settings['video_quality'], 'high'); ?>><?php _e('High', 'agentic-press'); ?></option>
                        <option value="ultra" <?php selected($settings['video_quality'], 'ultra'); ?>><?php _e('Ultra', 'agentic-press'); ?></option>
                    </select>
                    <p class="description"><?php _e('Quality level for generated videos.', 'agentic-press'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }

    public function affiliate_insights_callback() {
        $settings = $this->get_settings();
        ?>
        <table class="form-table">
            <tr>
                <th><label for="enable_affiliate_insights"><?php _e('Enable Affiliate Insights', 'agentic-press'); ?></label></th>
                <td>
                    <input type="checkbox" id="enable_affiliate_insights" name="<?php echo $this->option_name; ?>[enable_affiliate_insights]"
                           <?php checked($settings['enable_affiliate_insights']); ?>>
                    <p class="description"><?php _e('Enable AI-powered affiliate product suggestions.', 'agentic-press'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="max_affiliate_search_queries"><?php _e('Max Affiliate Searches', 'agentic-press'); ?></label></th>
                <td>
                    <input type="number" id="max_affiliate_search_queries" name="<?php echo $this->option_name; ?>[max_affiliate_search_queries]"
                           value="<?php echo esc_attr($settings['max_affiliate_search_queries']); ?>" min="1" max="10" class="small-text">
                    <p class="description"><?php _e('Maximum affiliate product searches per post.', 'agentic-press'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="results_per_product"><?php _e('Results Per Product', 'agentic-press'); ?></label></th>
                <td>
                    <input type="number" id="results_per_product" name="<?php echo $this->option_name; ?>[results_per_product]"
                           value="<?php echo esc_attr($settings['results_per_product']); ?>" min="1" max="5" class="small-text">
                    <p class="description"><?php _e('Number of affiliate results to return per search query.', 'agentic-press'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }

    public function google_search_api_callback() {
        $settings = $this->get_settings();
        ?>
        <table class="form-table">
            <tr>
                <th><label for="google_search_key"><?php _e('Google Search API Key', 'autopress'); ?></label></th>
                <td>
                    <input type="password" id="google_search_key" name="<?php echo $this->option_name; ?>[google_search_key]"
                           value="<?php echo esc_attr($settings['google_search_key']); ?>" class="regular-text">
                    <p class="description"><?php _e('Your Google Custom Search API key.', 'agentic-press'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="google_search_cx"><?php _e('Google Search Engine ID', 'autopress'); ?></label></th>
                <td>
                    <input type="text" id="google_search_cx" name="<?php echo $this->option_name; ?>[google_search_cx]"
                           value="<?php echo esc_attr($settings['google_search_cx']); ?>" class="regular-text">
                    <p class="description"><?php _e('Your Google Custom Search Engine ID (CX).', 'agentic-press'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="serpapi_key"><?php _e('SerpAPI Key', 'autopress'); ?></label></th>
                <td>
                    <input type="password" id="serpapi_key" name="<?php echo $this->option_name; ?>[serpapi_key]"
                           value="<?php echo esc_attr($settings['serpapi_key']); ?>" class="regular-text">
                    <p class="description"><?php _e('Your SerpAPI key for search functionality.', 'agentic-press'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="brave_search_key"><?php _e('Brave Search API Key', 'autopress'); ?></label></th>
                <td>
                    <input type="password" id="brave_search_key" name="<?php echo $this->option_name; ?>[brave_search_key]"
                           value="<?php echo esc_attr($settings['brave_search_key']); ?>" class="regular-text">
                    <p class="description"><?php _e('Your Brave Search API key.', 'agentic-press'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }

    public function debug_mode_callback() {
        $settings = $this->get_settings();
        ?>
        <input type="checkbox" id="debug_mode" name="<?php echo $this->option_name; ?>[debug_mode]"
               <?php checked($settings['debug_mode']); ?>>
        <p class="description"><?php _e('Enable debug logging for troubleshooting.', 'agentic-press'); ?></p>
        <?php
    }

    /**
     * Render the settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }

        ?>
        <div class="wrap">
            <h1><?php _e('Agentic Press Settings', 'agentic-press'); ?></h1>

            <form method="post" action="options.php">
                <?php
                settings_fields($this->option_name);
                do_settings_sections('autopress-settings');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }
}
